Leads API Helper
=================

Helping people work with FCA Leads API webservice (https://api.fcapoland.pl/leads).

Simple usage by example:

1. Include the library in your project:

   `$ composer require fca-poland/leads-api-helper`

2. Use it:
    ```php
    <?php
    require_once "vendor/autoload.php";

    $lead = new \FCAPoland\LeadsAPIHelper\Lead();
    // Attach logger if necessary
    $lead->setLogger(new \Psr\Log\NullLogger());
    $lead->setProperties([
        'type' => \FCAPoland\LeadsAPIHelper\Lead::TYPE_GETAQUOTE,
        'first_name' => 'Foo',
        'last_name' => 'Bar',
        'device' => \FCAPoland\LeadsAPIHelper\Lead::DEVICE_MOBILE,
        'privacy' => [
            \FCAPoland\LeadsAPIHelper\Lead::PRIVACY_TYPE_MARKETING => \FCAPoland\LeadsAPIHelper\Lead::PRIVACY_ACCEPTED,
            \FCAPoland\LeadsAPIHelper\Lead::PRIVACY_TYPE_PROFILING => \FCAPoland\LeadsAPIHelper\Lead::PRIVACY_REFUSED,
            \FCAPoland\LeadsAPIHelper\Lead::PRIVACY_TYPE_3RD_PARTY_MARKETING => \FCAPoland\LeadsAPIHelper\Lead::PRIVACY_ACCEPTED,
        ],
        // ...
    ]);

    $lead->setBrandId(\FCAPoland\LeadsAPIHelper\Lead::BRAND_JEEP);
    $lead->setOrigin('My landing page');
    // other `$lead->set...` comes here

    // setup API request:
    $api_url = 'https://api.fcapoland.pl';
    $api_user = 'johndoe';
    $api_pass = 'crackme';

    $api_request = new \FCAPoland\LeadsAPIHelper\Request\Create($api_url, $api_user, $api_pass);
    // or:
    $api_request = new \FCAPoland\LeadsAPIHelper\Request\Create();
    $api_request->setURL($api_url);
    $api_request->setUsername($api_user);
    $api_request->setPassword($api_pass);

    // Attach lead to the request
    $api_request->setLead($lead);

    try {
        $api_request->execute();
    } catch (\FCAPoland\LeadsAPIHelper\Exception $exception) {
        // Handle errors
        die($exception->getMessage());
    }
    $lead_id = $lead->getId();
    ```

Masterkey Finder
=================

Helping people find masterkey in FCA API webservice (https://api.fcapoland.pl/masterkey/find?).

Simple usage by example:

1. Use it:
    ```php
    <?php
    require_once "vendor/autoload.php";

    $masterkey_finder = new \FCAPoland\LeadsAPIHelper\MasterkeyFinder();
    // Attach logger if necessary
    $masterkey_finder->setLogger(new \Psr\Log\NullLogger());

    // Attach cacher if necessary for better performance. Cache must use CacheInterface
    // $masterkey_finder->setCache(CacheInterface $cache);

    // Required parameters to be set
    // Set brand
    $masterkey_finder->setBrand('00');
    // Set default app cid
    $masterkey_finder->setAppCid(7777);

    // Optional parameters to set
    // Set app config
    $masterkey_finder->setAppOid(5665);
    $masterkey_finder->setAppSource('standard');

    // Set params from quary
    $masterkey_finder->setCid(1235);
    $masterkey_finder->setOid(4566);
    $masterkey_finder->setSource('promo');

    // Get masterkey for type Test Drive
    $masterkey = $masterkey_finder
       ->find(\FCAPoland\LeadsAPIHelper\Lead::TYPE_TESTDRIVE)
       ->getMasterkey();

    // Get masterkey id
    $masterkey_id = $masterkey->getId();
    ```

Masterkey dabugger
-----------

1. Run debugger

    ```php
    $finder = new \FCAPoland\LeadsAPIHelper\MasterkeyFinder();
    $debugger = new \FCAPoland\LeadsAPIHelper\MasterkeyDebugger($finder);
     
    $debugger->addMkType(\FCAPoland\LeadsAPIHelper\Lead::TYPE_NEWSLETTER);
    $debugger->addMkType(\FCAPoland\LeadsAPIHelper\Lead::TYPE_GETAQUOTE);
    $debugger->addMkType(\FCAPoland\LeadsAPIHelper\Lead::TYPE_REQUEST_INFO);
     
    $debugger->run();
    ```
   
2. Add styles and html to end of html body.
    
    ```php
    $styles = $debugger->getCssFileContent();
    $html = $gebugger->getHTML();
    ```
    
3. Use Masterkey Debugger


Checking whether lead requires Call Center verification
=================

```php
$check = new LeadsAPIHelper\CheckCcVerification($api_url, $api_username, $api_password);
$result = $request->withId(123456); // or 'CCPOL1234' or 'PL1223456'

if ($result->isTrue()) {
  // ...
} elseif ($result->isFalse()) {
  // ...
} elseif ($result->isInvalidId()) {
  // ...
} elseif ($result->isNotFound()) {
  // ...
} elseif ($result->isServerError()) {
  // ...
} elseif ($result->isUnknown()) {
  // ...
} else {
  // This should not happen.
}

// Run another test with the same object, no need to create new one
$result_2 = $request->withId('PL1223456');
```


Development
-----------

> For Docker development environment see next section.

1. Instal development dependencies:

    `$ composer install --dev`

2. Checkout to development branch:

    `$ git checkout -b dev/feature-1`

3. Do the development (remember about tests!)
4. Sniff the code and detect mess:

    ```
    $ ./docker/sniff-the-code.sh
    $ ./docker/detect-mess.sh
    ```
5. Test:

    `$ ./test.sh`

   Leads API requires authorization. To pass authorization to the tests use following global constants (for example
   within your custom tests boostrap file, that sets constants and then calls regular, project's `tests/bootstrap.php`
   file saving other bootstraped settings):
    - `LEADS_API_USERNAME`
    - `LEADS_API_PASSWORD`

   Example:
   ```php
   <?php // contents of `my_bootstrap.php`
   define('LEADS_API_USERNAME', 'johndoe');
   define('LEADS_API_PASSWORD', 'breakmedown');
   require_once __DIR__ . '/tests/bootstrap.php';
   ```

   Also, one can override default API URL (which is `https://api.fcapoland.pl/leads`) by defining one's own using
   `LEADS_API_URL` constant the same way like for username and password:

   ```php
   <?php // contents of `my_bootstrap.php`
   define('LEADS_API_URL', 'http://localhost:8080');
   require_once __DIR__ . '/tests/bootstrap.php';
   ```

   And then:

       $ php ./vendor/bin/phpunit --bootstrap my_bootstrap.php

   or, if `test.sh` is availabe:

        $ ./test.sh --bootstrap my_bootstrap.php
        
   Note that there are currently two testsuites: `Unit` and `Functional`. The latter requires actual FCA API instance
   to be running and available. To use only single testsuite use `--testsuite` param like follows:

        $ ./test.sh --bootstrap my_bootstrap.php --testsuite Unit
   
6. Merge or create merge request.
7. Be happy PHP developer!

Docker
------

Docker utilities are available through `docker-compose`:

    $ docker-compose build

Then install Composer dependencies using:

    $ ./docker/[php72.sh|php74.sh] composer.phar install

When testing using Docker environment use `--docker` parameter added to the `test.sh` script, for example:

    $ ./docker/test.sh --bootstrap my_bootstrap.php
