<?php
declare(strict_types=1);

namespace FCAPoland\LeadsAPIHelper\Request;

use FCAPoland\LeadsAPIHelper\Exception\InvalidArgumentException;
use FCAPoland\LeadsAPIHelper\Exception\LeadsAPIHelperException;
use FCAPoland\LeadsAPIHelper\Request;

class CheckCcVerification extends Request
{
    public const ENDPOINT = '/leads/requires-cc-verification/<id>';

    /**
     * @var string|null
     */
    private $id = null;

    public function __construct(?string $url = null, ?string $api_username = null, ?string $api_password = null)
    {
        parent::__construct($url, $api_username, $api_password);

        // Replace endpoint path
        $this->setURL(str_replace(parse_url($this->url, PHP_URL_PATH), self::ENDPOINT, $this->url));
    }

    public function setID(string $id): void
    {
        // The only valid forms are:
        //   1. digits only ("12345")
        //   2. digits prefixed with "CCPOL"
        //   3. digits prefixed with "PL"
        if (!preg_match('/^(?:CCPOL|PL)?[0-9]+$/', $id)) {
            throw new InvalidArgumentException(sprintf(
                'Invalid lead ID: "%s", valid are: digits-only, digits prefixed with CCPOL, digits prefixed with PL',
                $id
            ));
        }

        $this->id = $id;
    }

    public function execute()
    {
        if (!isset($this->id)) {
            throw new LeadsAPIHelperException('Please set the lead ID first');
        }

        parent::execute();
    }

    protected function createURL(): string
    {
        // Replace "<id>" placeholder with current ID
        $url = parent::createURL();

        return strtr($url, ['<id>' => $this->id]);
    }
}
