<?php

namespace FCAPoland\ApiPrivacyHelper;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

/**
 * Class PrivacyPolicy
 * @package FCAPoland\ApiPrivacyHelper
 */
class PrivacyPolicy implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    const CACHE_KEY = PrivacyHelper::PRIVACY_HELPER_CACHE_KEY . ':privacy_policy:';
    const CACHE_KEY_BACKUP = PrivacyHelper::PRIVACY_HELPER_CACHE_KEY . ':privacy_policy_backup:';

    const CACHE_TTL = 60 * 60 * 3; // 3H
    const CACHE_TTL_BACKUP = 60 * 60 * 48; // 48H
    const API_URL = 'https://api.fcapoland.pl/shared-content/privacy/:HASH:';

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * @var string
     */
    private $general_privacy_policy_hash = '43a3bacb1a1268127c7b60f52b1f8675793007c0';

    /**
     * @param CacheInterface $cache
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
    }

    /**
     * Setter for policy hash
     * @param $hash
     */
    public function setPrivacyPolicyHash($hash)
    {
        $this->general_privacy_policy_hash = $hash;
    }

    /**
     * Getter for policy hash
     * @return string
     */
    public function getPrivacyPolicyHash()
    {
        return $this->general_privacy_policy_hash;
    }

    /**
     * Getter for policy hash
     * @return string
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function getPrivacyPolicyContent()
    {
        // Dwa przypadki cache jest o go nie ma. Tu rozpatrujemy gdy cache jest
        if ($this->cache != null) {
            // Próba pobrania polityki z cache
            $policy = $this->cache->get($this->getCacheKey());
            // Gdy w cache nie ma polityki
            if (!$policy) {
                // Próba pobrania z API i zapisu do cache i cache awaryjnego
                try {
                    $policy = $this->fetch();
                    $this->cache->set($this->getCacheKey(), $policy, self::CACHE_TTL);
                    $this->cache->set($this->getCacheKeyForBackup(), $policy, self::CACHE_KEY_BACKUP);
                } catch (Exception $e) {
                    // Gdy pobranie polityki z api nie jest możliwe próba pobrania z cache awaryjnego
                    $policy = $this->cache->get($this->getCacheKeyForBackup());
                    // Gdy nie ma polityki w cache awaryjnym błąd?
                    if (!$policy) {
                        throw new Exception($e->getMessage(), null, $e);
                    }
                    if ($this->logger instanceof LoggerInterface) {
                        $this->logger->warning('FCA Privacy Helper get privacy policy form backup cache for hash: :HASH:', [':HASH:' => $this->general_privacy_policy_hash]);
                    }
                }
            }
            return $policy;
        } else {
            // Próba pobrania polityki z api gdy pobierze zwraca treść w innym wypadku błąd
            try {
                $policy = $this->fetch();
                return $policy;
            } catch (Exception $e) {
                throw new Exception($e->getMessage(), null, $e);
            }
        }
    }

    /**
     * @return string
     */
    private function getCacheKey()
    {
        return self::CACHE_KEY . $this->general_privacy_policy_hash;
    }

    /**
     * @return mixed
     * @throws Exception
     */
    private function fetch()
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, strtr(static::API_URL, array(':HASH:' => $this->general_privacy_policy_hash)));
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, 'FCA Privacy Helper');
            $query = curl_exec($curl_handle);
            $httpcode = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $httpcode >= 400) {
                throw new Exception('Unable to get privacy data from FCA API for hash: ' . $this->general_privacy_policy_hash);
            }

            return $query;
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), null, $e);
        }
    }

    /**
     * @return string
     */
    private function getCacheKeyForBackup()
    {
        return self::CACHE_KEY_BACKUP . $this->general_privacy_policy_hash;
    }

    /**
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function clearCache()
    {
        if ($this->cache instanceof CacheInterface) {
            $this->cache->delete($this->getCacheKey());
            $this->cache->delete($this->getCacheKeyForBackup());
        }
    }
}
