<?php

namespace FCAPoland\ApiPrivacyHelper;

use FCAPoland\ApiPrivacyHelper\Disclaimer\Agreements;
use FCAPoland\ApiPrivacyHelper\Disclaimer\Information;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

class Disclaimer implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    const CACHE_KEY = PrivacyHelper::PRIVACY_HELPER_CACHE_KEY . '_disclaimer_';
    const CACHE_KEY_BACKUP = PrivacyHelper::PRIVACY_HELPER_CACHE_KEY . '_disclaimer_backup_';

    const CACHE_TTL = 10800; // 3H
    const CACHE_TTL_BACKUP = 172800; // 48H
    const API_URL = 'https://api.fcapoland.pl/shared-content/disclaimer/:DISCLAIMER_ID:.json';

    /**
     * @var int
     */
    private $disclaimer_id;

    /**
     * @var array
     */
    private $disclaimer;

    /**
     * @var Agreements
     */
    private $agreements;

    /**
     * @var Information
     */
    private $information;

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * Disclaimer constructor.
     * @param $disclaimer_id
     */
    public function __construct($disclaimer_id)
    {
        $this->disclaimer_id = $disclaimer_id;
        return $this;
    }

    /**
     * @return $this
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function getDisclaimer()
    {
        // Dwa przypadki cache jest o go nie ma. Tu rozpatrujemy gdy cache jest
        if ($this->cache != null) {
            // Próba pobrania zestawu z cache
            $disclaimer = $this->cache->get($this->getCacheKey());
            // Gdy w cache nie ma zestawu
            if (!$disclaimer) {
                // Próba pobrania z API i zapisu do cache i cache awaryjnego
                try {
                    $disclaimer = $this->fetch();
                    $this->cache->set($this->getCacheKey(), $disclaimer, self::CACHE_TTL);
                    $this->cache->set($this->getCacheKeyForBackup(), $disclaimer, self::CACHE_TTL_BACKUP);
                } catch (Exception $e) {
                    // Gdy pobranie polityki z api nie jest możliwe próba pobrania z cache awaryjnego
                    $disclaimer = $this->cache->get($this->getCacheKeyForBackup());
                    // Gdy nie ma zestawu zgód w cache awaryjnym błąd?
                    if (!$disclaimer) {
                        throw new Exception($e->getMessage(), null, $e);
                    }
                    if ($this->logger instanceof LoggerInterface) {
                        $this->logger->warning('FCA Privacy Helper get disclaimer form backup cache for id: :ID:', [':ID:' => $this->disclaimer_id]);
                    }
                }
            }
        } else {
            // Próba pobrania zestawu zgód z api gdy pobierze zapisuje do klasy w innym przypadku zwraca błąd
            try {
                $disclaimer = $this->fetch();
            } catch (Exception $e) {
                throw new Exception($e->getMessage(), null, $e);
            }
        }
        if (!$this->isJson($disclaimer)) {
            throw new Exception('Disclaimer json has invalid format!');
        }
        $this->disclaimer = json_decode($disclaimer, true);
        $this->agreements = new Agreements($this);
        $this->information = new Information($this);
        return $this;
    }

    /**
     * @return string
     */
    private function getCacheKey()
    {
        return self::CACHE_KEY . $this->disclaimer_id;
    }

    /**
     * @return mixed
     * @throws Exception
     */
    private function fetch()
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, strtr(static::API_URL, array(':DISCLAIMER_ID:' => $this->disclaimer_id)));
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, 'FCA Privacy Helper');
            $query = curl_exec($curl_handle);
            $httpcode = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $httpcode >= 400) {
                throw new Exception('Unable to get disclaimer from FCA API for id: ' . $this->disclaimer_id);
            }

            return $query;
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), null, $e);
        }
    }

    /**
     * @return string
     */
    private function getCacheKeyForBackup()
    {
        return self::CACHE_KEY_BACKUP . $this->disclaimer_id;
    }

    /**
     * @param $string
     * @return bool
     */
    private function isJson($string)
    {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }

    /**
     * @return int
     */
    public function getDisclaimerID()
    {
        return $this->disclaimer_id;
    }

    /**
     * @return array|mixed
     */
    public function getDisclaimerArray()
    {
        return $this->disclaimer;
    }

    /**
     * @return Agreements
     */
    public function getAgreements()
    {
        return $this->agreements;
    }

    /**
     * @return Information
     */
    public function getInformation()
    {
        return $this->information;
    }

    /**
     * @param CacheInterface $cache
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
    }

    /**
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function clearCache()
    {
        if ($this->cache instanceof CacheInterface) {
            $this->cache->delete($this->getCacheKey());
            $this->cache->delete($this->getCacheKeyForBackup());
        }
    }
}
