<?php

namespace FCAPoland\ApiPrivacyHelper;

use FCAPoland\ApiPrivacyHelper\Disclaimer\Agreements;
use FCAPoland\ApiPrivacyHelper\Disclaimer\Information;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

class Disclaimer implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    const CACHE_KEY = PrivacyHelper::PRIVACY_HELPER_CACHE_KEY . '_disclaimer_';

    const CACHE_TTL = 3600; // 1h
    const API_URL = 'https://api.fcapoland.pl/shared-content/disclaimer/:DISCLAIMER_ID:.json';

    /**
     * @var int
     */
    private $disclaimer_id;

    /**
     * @var array
     */
    private $disclaimer;

    /**
     * @var Agreements
     */
    private $agreements;

    /**
     * @var Information
     */
    private $information;

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * @var Storage
     */
    private $storage;


    /**
     * Disclaimer constructor.
     * @param $disclaimer_id
     * @throws Exception
     */
    public function __construct($disclaimer_id, $storage_file = null)
    {
        $this->disclaimer_id = $disclaimer_id;
        $this->storage = new Storage($storage_file);
        return $this;
    }

    /**
     * @return $this
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function getDisclaimer()
    {
        // Dwa przypadki cache jest o go nie ma. Tu rozpatrujemy gdy cache jest
        if ($this->cache != null) {
            // Próba pobrania zestawu z cache
            $disclaimer = $this->cache->get($this->getCacheKey());
            // Gdy w cache nie ma zestawu
            if (!$disclaimer) {
                try {
                    $disclaimer = $this->fetch();
                    $this->cache->set($this->getCacheKey(), $disclaimer, self::CACHE_TTL);
                    $this->storage->saveDisclaimer($this->getDisclaimerID(), $disclaimer);
                } catch (Exception $e) {
                    $disclaimer = $this->storage->getDisclaimer($this->getDisclaimerID());
                    if ($this->logger instanceof LoggerInterface) {
                        if (!$disclaimer) {
                            $this->logger->error('FCA Privacy Helper can not get disclaimer form storage for id: :ID:', [':ID:' => $this->getDisclaimerID()]);
                        } else {
                            $this->logger->warning('FCA Privacy Helper get disclaimer form storage for hash: :ID:', [':ID:' => $this->getDisclaimerID()]);
                        }
                    }
                }
            }
        } else {
            try {
                $disclaimer = $this->fetch();
                $this->storage->saveDisclaimer($this->getDisclaimerID(), $disclaimer);
            } catch (Exception $e) {
                $disclaimer = $this->storage->getDisclaimer($this->getDisclaimerID());
                if ($this->logger instanceof LoggerInterface) {
                    if (!$disclaimer) {
                        $this->logger->error('FCA Privacy Helper can not get disclaimer form storage for id: :ID:', [':ID:' => $this->getDisclaimerID()]);
                    } else {
                        $this->logger->warning('FCA Privacy Helper get disclaimer form storage for hash: :ID:', [':ID:' => $this->getDisclaimerID()]);
                    }
                }
            }
        }

        if ($disclaimer === false or !$this->isJson($disclaimer)) {
            throw new Exception('Disclaimer json has invalid format!');
        }
        $this->disclaimer = json_decode($disclaimer, true);
        $this->agreements = new Agreements($this);
        $this->information = new Information($this);
        return $this;
    }

    /**
     * @return string
     */
    private function getCacheKey()
    {
        return self::CACHE_KEY . $this->disclaimer_id;
    }

    /**
     * @return mixed
     * @throws Exception
     */
    private function fetch()
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, strtr(static::API_URL, array(':DISCLAIMER_ID:' => $this->disclaimer_id)));
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, 'FCA Privacy Helper');
            $query = curl_exec($curl_handle);
            $httpcode = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $httpcode >= 400) {
                throw new Exception('Unable to get disclaimer from FCA API for id: ' . $this->disclaimer_id);
            }

            return $query;
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), null, $e);
        }
    }

    /**
     * @param $string
     * @return bool
     */
    private function isJson($string)
    {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }

    /**
     * @return int
     */
    public function getDisclaimerID()
    {
        return $this->disclaimer_id;
    }

    /**
     * @return array|mixed
     */
    public function getDisclaimerArray()
    {
        return $this->disclaimer;
    }

    /**
     * @return Agreements
     */
    public function getAgreements()
    {
        return $this->agreements;
    }

    /**
     * @return Information
     */
    public function getInformation()
    {
        return $this->information;
    }

    /**
     * @param CacheInterface $cache
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
    }

    /**
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function clearCache()
    {
        if ($this->cache instanceof CacheInterface) {
            $this->cache->delete($this->getCacheKey());
        }
    }
}
