<?php
declare(strict_types=1);

namespace FCAPoland\ApiPrivacyHelper;

use FCAPoland\ApiPrivacyHelper\Exception\PrivacyHelperException;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;
use Psr\SimpleCache\InvalidArgumentException;

/**
 * Class PrivacyPolicy
 * @package FCAPoland\ApiPrivacyHelper
 */
class PrivacyPolicy implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    const CACHE_KEY = PrivacyHelper::PRIVACY_HELPER_CACHE_KEY . '_privacy_policy_';

    const CACHE_TTL = 3600; // 1H
    const API_URL = 'https://api.fcapoland.pl/shared-content/privacy/:HASH:';

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * Awaryjny magazyn konfiguracji polityki prywatności GDYBY API nie odpowiadało oraz zwykłe cache było niedostępne.
     * @var StorageInterface
     */
    private $storage;

    /**
     * Uwaga notka o prywatności jest pobierana z API, niepodanie hash'a powoduje pobranie domyślnej notki
     * @var string
     */
    private $general_privacy_policy_hash = '';

    /**
     * PrivacyPolicy constructor.
     * @param StorageInterface $storage
     */
    public function __construct(StorageInterface $storage)
    {
        $this->storage = $storage;
        return $this;
    }

    /**
     * @param CacheInterface $cache
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
    }

    /**
     * Setter for policy hash
     * @param $hash
     */
    public function setPrivacyPolicyHash($hash)
    {
        $this->general_privacy_policy_hash = $hash;
    }

    /**
     * Getter for policy hash
     * @return string
     */
    public function getPrivacyPolicyHash(): string
    {
        return $this->general_privacy_policy_hash;
    }

    /**
     * Getter for policy hash
     * @return string|null
     * @throws InvalidArgumentException
     */
    public function getPrivacyPolicyContent(): ?string
    {
        // Dwa przypadki cache jest o go nie ma. Tu rozpatrujemy gdy cache jest
        if ($this->cache != null) {
            // Próba pobrania polityki z cache
            $policy = $this->cache->get($this->getCacheKey());
            // Gdy w cache nie ma polityki
            if (!$policy) {
                try {
                    $policy = $this->fetch();
                    $this->cache->set($this->getCacheKey(), $policy, self::CACHE_TTL);
                    $this->storage->savePrivacyPolicy($this->getStorageKey(), $policy);
                } catch (PrivacyHelperException $e) {
                    $policy = $this->storage->getPrivacyPolicy($this->getStorageKey());
                    if ($this->logger instanceof LoggerInterface) {
                        if (!$policy) {
                            $this->logger->error('FCA Privacy Helper can not get privacy policy form storage for hash: :HASH:', [':HASH:' => $this->general_privacy_policy_hash]);
                        } else {
                            $this->logger->warning('FCA Privacy Helper get privacy policy form storage for hash: :HASH:', [':HASH:' => $this->general_privacy_policy_hash]);
                        }
                    }
                }
            }
        } else {
            try {
                $policy = $this->fetch();
                $this->storage->savePrivacyPolicy($this->getStorageKey(), $policy);
                return $policy;
            } catch (PrivacyHelperException $e) {
                $policy = $this->storage->getPrivacyPolicy($this->getStorageKey());
                if ($this->logger instanceof LoggerInterface) {
                    if (!$policy) {
                        $this->logger->error('FCA Privacy Helper can not get privacy policy form storage for hash: :HASH:', [':HASH:' => $this->general_privacy_policy_hash]);
                    } else {
                        $this->logger->warning('FCA Privacy Helper get privacy policy form storage for hash: :HASH:', [':HASH:' => $this->general_privacy_policy_hash]);
                    }
                }
            }
        }
        return $policy;
    }

    /**
     * @return string
     */
    private function getCacheKey(): string
    {
        return md5(self::CACHE_KEY . $this->general_privacy_policy_hash);
    }

    /**
     * @return string
     */
    private function getStorageKey(): string
    {
        return !empty($this->general_privacy_policy_hash) ? $this->general_privacy_policy_hash : 'default';
    }

    /**
     * @return string
     * @throws PrivacyHelperException
     */
    private function fetch(): string
    {
        try {
            $curl_handle = curl_init();
            curl_setopt(
                $curl_handle,
                CURLOPT_URL,
                strtr(static::API_URL, array(':HASH:' => $this->general_privacy_policy_hash))
            );
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, 'FCA Privacy Helper');
            $query = curl_exec($curl_handle);
            $http_code = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $http_code >= 400) {
                throw new PrivacyHelperException(
                    'Unable to get privacy data from FCA API for hash: ' . $this->general_privacy_policy_hash
                );
            }

            return $query;
        } catch (PrivacyHelperException $e) {
            throw new PrivacyHelperException($e->getMessage(), 0, $e);
        }
    }

    /**
     * @throws InvalidArgumentException
     */
    public function clearCache()
    {
        if ($this->cache instanceof CacheInterface) {
            $this->cache->delete($this->getCacheKey());
        }
    }

    /**
     * @param StorageInterface $storage
     */
    public function setStorage(StorageInterface $storage)
    {
        $this->storage = $storage;
    }
}
