<?php
declare(strict_types=1);

namespace FCAPoland\ApiPrivacyHelper\Cache;

use FCAPoland\ApiPrivacyHelper\Exception\CachePrivacyHelperException;
use Psr\SimpleCache\CacheInterface;

/**
 * Class MemoryCache
 *
 * Simple in-memory cache pool mainly for testing purposes.
 * It is flushed at the end of the script.
 *
 * @package FCAPoland\ApiPrivacyHelper\Cache
 */
final class MemoryCache implements CacheInterface
{
    /**
     * @var array
     */
    private $storage;

    /**
     * MemoryCache constructor.
     * @param array $storage
     */
    public function __construct(array &$storage = [])
    {
        $this->storage = &$storage;
    }

    /**
     * @inheritdoc
     * @throws CachePrivacyHelperException
     */
    public function getMultiple($keys, $default = null)
    {
        foreach ($keys as $key) {
            yield $this->get($key, $default);
        }
    }

    /**
     * @inheritdoc
     * @throws CachePrivacyHelperException
     */
    public function get($key, $default = null)
    {
        if (!is_string($key)) {
            throw new CachePrivacyHelperException('Cache key must be a string!');
        }

        if (array_key_exists($key, $this->storage)) {
            return $this->storage[$key];
        }

        return $default;
    }

    /**
     * @inheritdoc
     * @throws CachePrivacyHelperException
     */
    public function has($key): bool
    {
        if (!is_string($key)) {
            throw new CachePrivacyHelperException('Cache key must be a string!');
        }

        return array_key_exists($key, $this->storage);
    }

    /**
     * @inheritdoc
     */
    public function clear(): bool
    {
        $this->storage = [];
        return true;
    }

    /**
     * @inheritdoc
     * @throws CachePrivacyHelperException
     */
    public function deleteMultiple($keys): bool
    {
        foreach ($keys as $key) {
            $this->delete($key);
        }

        return true;
    }

    /**
     * @inheritdoc
     * @throws CachePrivacyHelperException
     */
    public function delete($key): bool
    {
        if (!is_string($key)) {
            throw new CachePrivacyHelperException('Cache key must be a string!');
        }

        unset($this->storage[$key]);

        return true;
    }

    /**
     * @inheritdoc
     */
    public function setMultiple($values, $ttl = null): bool
    {
        foreach ($values as $key => $value) {
            $this->set($key, $value, $ttl);
        }
        return true;
    }

    /**
     * @inheritdoc
     */
    public function set($key, $value, $ttl = null)
    {
        $this->storage[$key] = $value;
    }
}
