<?php
declare(strict_types=1);

namespace FCAPoland\ApiPrivacyHelper;

use FCAPoland\ApiPrivacyHelper\Disclaimer\Agreements;
use FCAPoland\ApiPrivacyHelper\Disclaimer\Information;
use FCAPoland\ApiPrivacyHelper\Exception\PrivacyHelperException;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;
use Psr\SimpleCache\InvalidArgumentException;

/**
 * Class Disclaimer
 * @package FCAPoland\ApiPrivacyHelper
 */
class Disclaimer implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    const CACHE_KEY = PrivacyHelper::PRIVACY_HELPER_CACHE_KEY . '_disclaimer_';

    const CACHE_TTL = 3600; // 1h

    const API_URL = 'https://api.fcapoland.pl/shared-content/disclaimer/:DISCLAIMER_ID:.json';

    /**
     * @var int
     */
    private $disclaimer_id;

    /**
     * @var array
     */
    private $disclaimer;

    /**
     * @var Agreements
     */
    private $agreements;

    /**
     * @var Information
     */
    private $information;

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * @var Storage
     */
    private $storage;


    /**
     * Disclaimer constructor.
     * @param $disclaimer_id
     * @param StorageInterface $storage
     */
    public function __construct($disclaimer_id, StorageInterface $storage)
    {
        $this->disclaimer_id = $disclaimer_id;
        $this->storage = $storage;
        return $this;
    }

    /**
     * @return $this
     * @throws PrivacyHelperException
     * @throws InvalidArgumentException
     * @throwss Exception
     * @throwss InvalidArgumentException
     */
    public function getDisclaimer(): self
    {
        // Dwa przypadki cache jest o go nie ma. Tu rozpatrujemy, gdy cache jest
        if ($this->cache != null) {
            // Próba pobrania zestawu z cache
            $disclaimer = $this->cache->get($this->getCacheHash());
            // Gdy w cache nie ma zestawu
            if (!$disclaimer) {
                try {
                    $disclaimer = $this->fetch();
                    $this->cache->set($this->getCacheHash(), $disclaimer, self::CACHE_TTL);
                    $this->storage->saveDisclaimer($this->getDisclaimerID(), $disclaimer);
                } catch (PrivacyHelperException $e) {
                    $disclaimer = $this->storage->getDisclaimer($this->getDisclaimerID());
                    if ($this->logger instanceof LoggerInterface) {
                        if (!$disclaimer) {
                            $this->logger->error('FCA Privacy Helper can not get disclaimer form storage for id: :ID:', [':ID:' => $this->getDisclaimerID()]);
                        } else {
                            $this->logger->warning('FCA Privacy Helper get disclaimer form storage for hash: :ID:', [':ID:' => $this->getDisclaimerID()]);
                        }
                    }
                }
            }
        } else {
            try {
                $disclaimer = $this->fetch();
                $this->storage->saveDisclaimer($this->getDisclaimerID(), $disclaimer);
            } catch (PrivacyHelperException $e) {
                $disclaimer = $this->storage->getDisclaimer($this->getDisclaimerID());
                if ($this->logger instanceof LoggerInterface) {
                    if (!$disclaimer) {
                        $this->logger->error('FCA Privacy Helper can not get disclaimer form storage for id: :ID:', [':ID:' => $this->getDisclaimerID()]);
                    } else {
                        $this->logger->warning('FCA Privacy Helper get disclaimer form storage for hash: :ID:', [':ID:' => $this->getDisclaimerID()]);
                    }
                }
            }
        }

        if ($disclaimer === false or $disclaimer === null or !$this->isJson($disclaimer)) {
            throw new PrivacyHelperException('Disclaimer json has invalid format!');
        }
        $this->disclaimer = @json_decode($disclaimer, true);
        $this->agreements = new Agreements($this);
        $this->information = new Information($this);
        return $this;
    }

    /**
     * @return string
     */
    private function getCacheHash(): string
    {
        return md5(self::CACHE_KEY . $this->disclaimer_id);
    }

    /**
     * @return string
     */
    private function getStorageCacheKey(): string
    {
        return !empty($this->disclaimer_id) ? (string) $this->disclaimer_id : 'default';
    }

    /**
     * @return mixed
     * @throws PrivacyHelperException
     */
    private function fetch(): string
    {
        $curl_handle = curl_init();
        curl_setopt($curl_handle, CURLOPT_URL, strtr(static::API_URL, array(':DISCLAIMER_ID:' => $this->disclaimer_id)));
        curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
        curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl_handle, CURLOPT_USERAGENT, 'FCA Privacy Helper');
        $query = curl_exec($curl_handle);
        $http_code = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
        curl_close($curl_handle);

        if ($query === false or $http_code >= 400) {
            throw new PrivacyHelperException('Unable to get disclaimer from FCA API for id: ' . $this->disclaimer_id);
        }

        return $query;
    }

    /**
     * @param $string
     * @return bool
     */
    private function isJson($string): bool
    {
        @json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }

    /**
     * @return int
     */
    public function getDisclaimerID(): int
    {
        return $this->disclaimer_id;
    }

    /**
     * @return array|mixed
     */
    public function getDisclaimerArray(): array
    {
        return $this->disclaimer;
    }

    /**
     * @return Agreements
     */
    public function getAgreements(): Agreements
    {
        return $this->agreements;
    }

    /**
     * @return Information
     */
    public function getInformation(): Information
    {
        return $this->information;
    }

    /**
     * @param CacheInterface $cache
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
    }

    /**
     * @throws InvalidArgumentException
     */
    public function clearCache()
    {
        if ($this->cache instanceof CacheInterface) {
            $this->cache->delete($this->getCacheHash());
        }
    }

    /**
     * @param StorageInterface $storage
     */
    public function setStorage(StorageInterface $storage)
    {
        $this->storage = $storage;
    }
}
