<?php
declare(strict_types=1);

namespace FCAPoland\ApiPrivacyHelper;

use FCAPoland\ApiPrivacyHelper\Exception\PrivacyHelperException;

/**
 * Class Storage
 * @package FCAPoland\ApiPrivacyHelper
 */
class Storage implements StorageInterface
{
    const STORAGE_FILE = 'helpers_cache/privacy_helper_storage';

    /** @var array */
    private $storage;

    /** @var string */
    private $storage_file;

    /**
     * Storage constructor.
     * @param null $storage_file
     * @throws PrivacyHelperException
     */
    public function __construct($storage_file = null)
    {
        $this->storage_file = $storage_file ?? self::STORAGE_FILE;
        if (!file_exists($this->storage_file)) {
            $this->createStorageFile();
        }
        $this->readStorage();
    }

    /**
     * @param $id
     * @param $data
     * @throws PrivacyHelperException
     */
    public function saveDisclaimer($id, $data)
    {
        $this->readStorage();
        $this->storage['disclaimer'][$id] = $data;
        $this->saveStorage();
    }

    /**
     * @param $id
     * @return mixed
     */
    public function getDisclaimer($id)
    {
        return $this->storage['disclaimer'][$id] ?? null;
    }

    /**
     * @param $hash
     * @param $data
     * @throws PrivacyHelperException
     */
    public function savePrivacyPolicy($hash, $data)
    {
        $this->readStorage();
        $this->storage['privacy_policy'][$hash] = $data;
        $this->saveStorage();
    }

    /**
     * @param string $hash
     * @return mixed
     */
    public function getPrivacyPolicy(string $hash): ?string
    {
        return $this->storage['privacy_policy'][$hash] ?? null;
    }

    /**
     * @throws PrivacyHelperException
     */
    private function readStorage()
    {
        $storage_file = file_get_contents($this->storage_file);
        if ($storage_file === false) {
            throw new PrivacyHelperException('Can not read storage file');
        }
        if (strlen($storage_file) == 0) {
            $this->storage = [];
        }
        $this->storage = json_decode($storage_file, true);
    }

    /**
     * @param string $key
     * @param $data
     * @throws PrivacyHelperException
     */
    public function saveToStorage(string $key, $data)
    {
        $this->readStorage();
        $this->storage[$key] = $data;
        $this->saveStorage();
    }

    /**
     * @param string $key
     * @return mixed|null
     * @throws PrivacyHelperException
     */
    public function readFromStorage(string $key)
    {
        $this->readStorage();
        return $this->storage[$key] ?? null;
    }

    /**
     * Save to storage file
     */
    private function saveStorage()
    {
        if (!is_array($this->storage)) {
            $this->storage = [];
        }
        $json_storage = @json_encode($this->storage);
        file_put_contents($this->storage_file, $json_storage, LOCK_EX);
    }

    /**
     * Create storage file if not exists
     */
    private function createStorageFile()
    {
        if (!file_exists($this->storage_file)) {
            $storage_file = fopen($this->storage_file, 'w');
            fclose($storage_file);
            chmod($this->storage_file, 0664);
        }
    }

    /**
     * Remove storage file and clear storage
     */
    public function removeStorage()
    {
        if (file_exists($this->storage_file)) {
            unlink($this->storage_file);
            $this->storage = [];
        }
    }

    public function saveDealers(string $form_and_brand_id, $data)
    {
        // TODO: Implement saveDealers() method.
    }

    public function getDealers(string $form_and_brand_id)
    {
        // TODO: Implement getDealers() method.
    }
}
