<?php
/**
 * Created by PhpStorm.
 * Damian Banaszkiewicz <dbanaszkiewicz@unicorn.net.pl>
 * 30.08.18 13:55
 */

namespace FCA\StockApi\Tests\Unit\Collection\Filter;

use FCA\StockApi\Collection\Filter\Builder\Fields;
use FCA\StockApi\Collection\Filter\Stock;
use FCA\StockApi\Exception\ApiException;
use PHPUnit\Framework\TestCase;

class StockTest extends TestCase
{
    /**
     * @throws ApiException
     */
    public function testGetFilter()
    {
        $stock = new Stock();
        $result = $stock->getFilter();
        $this->assertEquals([], $result);

        $stock = new Stock();
        $stock->setEnginePowerTo(104);
        $stock->setEnginePowerFrom(90);
        $result = $stock->getFilter();

        $returnArray = [
            '$and' => [
                [
                    Fields::ENGINE_POWER => [
                        '$lte' => 104
                    ]
                ],
                [
                    Fields::ENGINE_POWER => [
                        '$gte' => 90
                    ]
                ]
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetEnginePowerTo()
    {
        $value = 55;

        $stock = new Stock();
        $stock->setEnginePowerTo($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::ENGINE_POWER => [
                '$lte' => $value
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetEnginePowerFrom()
    {
        $value = 55;

        $stock = new Stock();
        $stock->setEnginePowerFrom($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::ENGINE_POWER => [
                '$gte' => $value
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }


    /**
     * @throws ApiException
     */
    public function testSetPriceTypeWithInvalidValue()
    {
        $this->expectException(ApiException::class);
        $stock = new Stock();
        $stock->setPriceType('invalidType');
    }

    /**
     * @throws ApiException
     */
    public function testSetFuelType()
    {
        $value = "Benzyna";

        $stock = new Stock();
        $stock->setFuelType($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::FUEL_TYPE => [
                '$in' => [$value]
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetModel()
    {
        $value = "357HB";

        $stock = new Stock();
        $stock->setModel($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::MODEL_COMMERCIAL_CODE => [
                '$in' => [$value]
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetInteriorType()
    {
        $value = "Skóra";

        $stock = new Stock();
        $stock->setInteriorType($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::INTERIOR_TYPE => [
                '$in' => [$value]
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetPriceFrom()
    {
        $value = 65464.00;

        $stock = new Stock();
        $stock->setPriceFrom($value);
        $stock->setPriceType(Fields::FINAL_PRICE_BRUTTO);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::FINAL_PRICE_BRUTTO => [
                '$gte' => $value
            ]
        ];

        $this->assertEquals($returnArray, $result);

        $stock = new Stock();
        $stock->setPriceFrom($value);
        $this->expectException(ApiException::class);
        $stock->getFilter();
    }

    /**
     * @throws ApiException
     */
    public function testSetPriceTo()
    {
        $value = 65464.00;

        $stock = new Stock();
        $stock->setPriceTo($value);
        $stock->setPriceType(Fields::FINAL_PRICE_BRUTTO);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::FINAL_PRICE_BRUTTO => [
                '$lte' => $value
            ]
        ];

        $this->assertEquals($returnArray, $result);

        $stock = new Stock();
        $stock->setPriceTo($value);
        $this->expectException(ApiException::class);
        $stock->getFilter();
    }

    /**
     * @throws ApiException
     */
    public function testSetProductionYear()
    {
        $value = 2018;

        $stock = new Stock();
        $stock->setProductionYear($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::PRODUCTION_YEAR => [
                '$in' => [$value]
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetGearboxType()
    {
        $value = "Automatyczna";

        $stock = new Stock();
        $stock->setGearboxType($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::GEARBOX_TYPE => [
                '$in' => [$value]
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetBrandCode()
    {
        $value = "00";

        $stock = new Stock();
        $stock->setBrandCode($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::BRAND_CODE => $value
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetDriveType()
    {
        $value = "Napęd na przednie koła";

        $stock = new Stock();
        $stock->setDriveType($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::DRIVE_TYPE => [
                '$in' => [$value]
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetColor()
    {
        $value = "WHITE";

        $stock = new Stock();
        $stock->setColor($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::COLOR_CATEGORY => [
                '$in' => [$value]
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testSetEquipmentCodes()
    {
        $value = "BNG";

        $stock = new Stock();
        $stock->setEquipmentCode($value);
        $result = $stock->getFilter();

        $returnArray = [
            Fields::EQUIPMENT_CODE => [
                '$in' => [$value]
            ]
        ];

        $this->assertEquals($returnArray, $result);
    }

    /**
     * @throws ApiException
     */
    public function testNearByCoordinates()
    {
        $stock = new Stock();
        $stock->setLatitude(12.54465);
        $stock->setLongitude(45.55478);
        $stock->setMaxDistance(10000);

        $expectedArray = [
            Fields::DEALER_COORDINATES => [
                '$near' => [
                    '$geometry' => ['type' => "Point", 'coordinates' => [45.55478, 12.54465]],
                    '$maxDistance' => 10000
                ]
            ]
        ];

        $this->assertEquals($expectedArray, $stock->getFilter());

        //===========================================================================================

        $stock = new Stock();
        //$stock->setLatitude(12.54465);
        $stock->setLongitude(45.55478);
        $stock->setMaxDistance(10000);


        $this->assertEquals([], $stock->getFilter());

        //===========================================================================================

        $stock = new Stock();
        $stock->setLatitude(12.54465);
        //$stock->setLongitude(45.55478);
        $stock->setMaxDistance(10000);


        $this->assertEquals([], $stock->getFilter());

        //===========================================================================================

        $stock = new Stock();
        $stock->setLatitude(12.54465);
        $stock->setLongitude(45.55478);
        //$stock->setMaxDistance(10000);


        $this->assertEquals([], $stock->getFilter());
    }

    /**
     * @throws ApiException
     */
    public function testSetLatitudeWithBadValue()
    {
        $stock = new Stock();
        $this->expectException(ApiException::class);
        $stock->setLongitude("45.55478");
    }

    /**
     * @throws ApiException
     */
    public function testSetLongitudeWithBadValue()
    {
        $stock = new Stock();
        $this->expectException(ApiException::class);
        $stock->setLongitude("45.55478");
    }

    /**
     * @throws ApiException
     */
    public function testSetMaxDistanceWithBadValue()
    {
        $stock = new Stock();
        $this->expectException(ApiException::class);
        $stock->setLongitude("45.55478");
    }
}
