<?php

namespace FCA\StockApi\Collection\Filter;

use FCA\StockApi\Collection\Filter\Builder\Validator;
use FCA\StockApi\Exception\ApiException;

/**
 * Class Builder
 * @package FCA\StockApi\Collection\Filter
 */
class Builder
{
    /**
     * @param array ...$filters
     * @return array
     * @throws ApiException
     */
    public function orWhere(array ...$filters)
    {
        foreach ($filters as $filter) {
            Validator::validateFilter($filter);
        }

        if (count($filters) > 1) {
            return [
                '$or' => $filters
            ];
        } elseif (count($filters) === 1) {
            return $filters[0];
        } else {
            return [];
        }
    }

    /**
     * @param array ...$filters
     * @return array
     * @throws ApiException
     */
    public function andWhere(array ...$filters)
    {
        foreach ($filters as $filter) {
            Validator::validateFilter($filter);
        }

        if (count($filters) > 1) {
            return [
                '$and' => $filters
            ];
        } elseif (count($filters) === 1) {
            return $filters[0];
        } else {
            return [];
        }
    }

    /**
     * @param array $filter
     * @return array
     * @throws ApiException
     */
    public function not(array $filter)
    {
        Validator::validateExpression($filter);

        $key = array_keys($filter)[0];
        return [
            $key => [
                '$not' => $filter[$key]
            ]
        ];
    }

    /**
     * @param $field
     * @param $operator (Valid values: =, !=, <, <=, >, >=, in, !in)
     * @param $value
     * @return array
     * @throws ApiException
     */
    public function is($field, $operator, $value)
    {
        // check is field valid
        if (!Validator::isValidField($field)) {
            throw new ApiException('Invalid field "' . $field . '"');
        }

        // check is value type valid
        if ($operator === 'in' or $operator === '!in') {
            if (!is_array($value)) {
                throw new ApiException('Value must be an array!');
            }

            foreach ($value as $val) {
                if (!Validator::isValidValue($field, $val)) {
                    throw new ApiException('Invalid value type ("' . $field . '")');
                }
            }
        } else {
            if (!Validator::isValidValue($field, $value)) {
                throw new ApiException('Invalid value type ("' . $field . '")');
            }
        }

        // return array or throw exception when operator is not valid
        if ($operator === '<=') {
            return [$field => ['$lte' => $value]];
        } elseif ($operator === '<') {
            return [$field => ['$lt' => $value]];
        } elseif ($operator === '>=') {
            return [$field => ['$gte' => $value]];
        } elseif ($operator === '>') {
            return [$field => ['$gt' => $value]];
        } elseif ($operator === '=') {
            return [$field => $value];
        } elseif ($operator === '!=') {
            return [$field => ['$ne' => $value]];
        } elseif ($operator === 'in') {
            return [$field => ['$in' => $value]];
        } elseif ($operator === '!in') {
            return [$field => ['$nin' => $value]];
        } else {
            throw new ApiException('Invalid operator! (Valid values: =, !=, <, <=, >, >=, in, !in)');
        }
    }
    /**
     * @param $field
     * @param $operator (Valid values: =, !=, <, <=, >, >=, in, !in)
     * @param $value
     * @return array
     * @throws ApiException
     */
    public function isNear($field, $longitude, $latitude, $maxDistance)
    {
        $filter = [
            $field => [
                '$near' => [
                    '$geometry' => [
                        'type' => 'Point',
                        'coordinates' => [$longitude, $latitude]
                    ],
                    '$maxDistance' => $maxDistance
                ]
            ]
        ];

        Validator::validateFilter($filter);

        return $filter;
    }
}
