<?php

namespace FCA\StockApi\Document;

use FCA\StockApi\Document\Offer\Brand;
use FCA\StockApi\Document\Offer\Color;
use FCA\StockApi\Document\Offer\Dealer;
use FCA\StockApi\Document\Offer\Equipment;
use FCA\StockApi\Document\Offer\Interior;
use FCA\StockApi\Document\Offer\Model;
use FCA\StockApi\Document\Offer\Price;
use FCA\StockApi\Document\Offer\Series;
use FCA\StockApi\Document\Offer\TechnicalData;
use FCA\StockApi\Document\Offer\Version;
use FCA\StockApi\Exception\ApiException;

class Offer extends Base
{
    /**
     * @var int
     */
    protected $id;

    /**
     * @var Brand
     */
    protected $brand;

    /**
     * @var Model
     */
    protected $model;

    /**
     * @var Version
     */
    protected $version;

    /**
     * @var Series
     */
    protected $series;

    /**
     * @var Dealer
     */
    protected $dealer;

    /**
     * @var string
     */
    protected $status;

    /**
     * @var Price
     */
    protected $price;

    /**
     * @var boolean
     */
    protected $isDemo;

    /**
     * @var int
     */
    protected $productionYear;

    /**
     * @var string
     */
    protected $productionDate;

    /**
     * @var string
     */
    protected $sincom;

    /**
     * @var string[]
     */
    protected $stamps;

    /**
     * @var Color
     */
    protected $color;

    /**
     * @var Interior
     */
    protected $interior;

    /**
     * @var null|string
     */
    protected $description;

    /**
     * @var string
     */
    protected $serialNo;

    /**
     * @var string[]
     */
    protected $labels;

    /**
     * @var array
     */
    protected $images;

    /**
     * @var Equipment[]
     */
    protected $equipment = [];

    /**
     * @var TechnicalData
     */
    protected $technicalData;

    /**
     * @var array
     */
    protected $groups = [];

    /**
     * Offer constructor.
     * @param $offer
     * @throws ApiException
     */
    public function __construct($offer)
    {
        $requiredKeys = ['vehicle_id', 'brand', 'model', 'version', 'series', 'dealer', 'status', 'price',
            'is_demo', 'year_of_production', 'date_of_production', 'color', 'interior', 'description', 'serial_no',
            'images', 'labels', 'equipment', 'technical_data', 'sincom', 'stamps'];
        $this->checkKeys($requiredKeys, $offer);

        $this->id = $offer['vehicle_id'];
        $this->brand = new Brand((array)$offer['brand']);
        $this->model = new Model((array)$offer['model']);
        $this->version = new Version((array)$offer['version']);
        $this->series = new Series((array)$offer['series']);

        if ($offer['dealer'] !== null) {
            $this->dealer = new Dealer((array)$offer['dealer']);
        } else {
            $this->dealer = null;
        }

        $this->status = $offer['status'];
        $this->price = new Price((array)$offer['price']);
        $this->isDemo = $offer['is_demo'];
        $this->productionYear = $offer['year_of_production'];
        $this->productionDate = $offer['date_of_production'];
        $this->sincom = $offer['sincom'];
        $this->stamps = $offer['stamps'];
        $this->color = new Color((array)$offer['color']);
        $this->interior = new Interior((array)$offer ['interior']);
        $this->description = $offer['description'];
        $this->serialNo = $offer['serial_no'];
        $this->images = (array)$offer['images'];
        $this->labels = (array)$offer['labels'];

        foreach ($offer['equipment'] as $equipment) {
            $this->equipment[] = new Equipment((array)$equipment);
        }

        $this->technicalData = new TechnicalData((array)$offer['technical_data']);

        if (isset($offer['groups'])) {
            $this->groups = (array)$offer['groups'];
        }
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return Brand
     */
    public function getBrand()
    {
        return $this->brand;
    }

    /**
     * @return Model
     */
    public function getModel()
    {
        return $this->model;
    }

    /**
     * @return Version
     */
    public function getVersion()
    {
        return $this->version;
    }

    /**
     * @return Series
     */
    public function getSeries()
    {
        return $this->series;
    }

    /**
     * @return Dealer
     */
    public function getDealer()
    {
        return $this->dealer;
    }

    /**
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * @return Price
     */
    public function getPrice()
    {
        return $this->price;
    }

    /**
     * @return bool
     */
    public function getIsDemo()
    {
        return $this->isDemo;
    }

    /**
     * @return int
     */
    public function getProductionYear()
    {
        return $this->productionYear;
    }

    /**
     * @return string
     */
    public function getProductionDate()
    {
        return $this->productionDate;
    }

    /**
     * @return string
     */
    public function getSincom()
    {
        return $this->sincom;
    }

    /**
     * @return string[]
     */
    public function getStamps()
    {
        return $this->stamps;
    }

    /**
     * @return Color
     */
    public function getColor()
    {
        return $this->color;
    }

    /**
     * @return Interior
     */
    public function getInterior()
    {
        return $this->interior;
    }

    /**
     * @return null|string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @return string
     */
    public function getSerialNo()
    {
        return $this->serialNo;
    }

    /**
     * @return array
     */
    public function getImages()
    {
        return $this->images;
    }

    /**
     * @return string[]
     */
    public function getLabels()
    {
        return $this->labels;
    }

    /**
     * @return Equipment[]
     */
    public function getEquipment()
    {
        return $this->equipment;
    }

    /**
     * @return TechnicalData
     */
    public function getTechnicalData()
    {
        return $this->technicalData;
    }

    /**
     * @return array
     */
    public function getGroups()
    {
        return $this->groups;
    }

    private function getEquipmentArray()
    {
        $array = [];

        foreach ($this->getEquipment() as $equipment) {
            $array[] = $equipment->toArray();
        }

        return $array;
    }

    public function toArray()
    {
        return [
            'id' => $this->getId(),
            'brand' => $this->getBrand()->toArray(),
            'model' => $this->getModel()->toArray(),
            'version' => $this->getVersion()->toArray(),
            'series' => $this->getSeries()->toArray(),
            'dealer' => $this->getDealer()->toArray(),
            'status' => $this->getStatus(),
            'price' => $this->getPrice()->toArray(),
            'isDemo' => $this->getIsDemo(),
            'productionYear' => $this->getProductionYear(),
            'productionDate' => $this->getProductionDate(),
            'sincom' => $this->getSincom(),
            'stamps' => $this->getStamps(),
            'color' => $this->getColor()->toArray(),
            'interior' => $this->getInterior()->toArray(),
            'description' => $this->getDescription(),
            'serialNo' => $this->getSerialNo(),
            'images' => $this->getImages(),
            'labels' => $this->getLabels(),
            'equipment' => $this->getEquipmentArray(),
            'technicalData' => $this->getTechnicalData()->toArray(),
            'groups' => $this->getGroups(),
        ];
    }
}
