<?php
declare(strict_types=1);

namespace FCA\StockApi\Collection;

use FCA\StockApi\Api;
use FCA\StockApi\Collection\Filter\Builder\Fields;
use FCA\StockApi\Collection\Filter\Builder\Validator;
use FCA\StockApi\Collection\Sort\Sort;
use FCA\StockApi\Document\DistinctValue;
use FCA\StockApi\Exception\ApiException;

class Offer
{
    /**
     * @var \MongoDB\Collection
     */
    protected $offerCollection;

    /**
     * Offer constructor.
     * @param Api $api
     * @throws ApiException
     */
    public function __construct(Api $api)
    {
        if ($api->getCollection() === null) {
            throw new ApiException('Database collection is missing!');
        }

        $this->offerCollection = $api->getCollection();
    }

    /**
     * @param array $filters
     * @param array|null $sort
     * @param int|null $limit
     * @param int|null $skip
     * @return \FCA\StockApi\Document\Offer[]
     * @throws ApiException
     */
    public function find(array $filters = [], ?array $sort = null, ?int $limit = null, ?int $skip = null): array
    {
        Validator::validateFilter($filters);

        $options = [
            'collation' => ['locale' => 'pl'],
        ];

        $pipeline = $this->getBasicPipelines($filters, $sort);

        if ($sort === ['random' => true]) {
            if ($limit === null) {
                throw new ApiException('Limit must be defined in random find!');
            }
            $pipeline[] = [
                '$sample' => [
                    'size' => $limit
                ]
            ];
        } else {
            if ($skip) {
                $pipeline[] = [
                    '$skip' => $skip
                ];
            }
            if ($limit) {
                $pipeline[] = [
                    '$limit' => $limit
                ];
            }
        }

        $docs = $this->offerCollection->aggregate(
            $pipeline,
            $options
        );

        $offers = [];

        foreach ($docs as $doc) {
            $offers[] = new \FCA\StockApi\Document\Offer((array)$doc);
        }

        $docs = null;
        unset($docs);

        return $offers;
    }

    private function getBasicPipelines(array $filters = [], ?array $sort = null): array
    {
        $pipeline = [];

        $brand_code = $this->getBrandCodeFromFiltersArray($filters);

        $pipeline[] = $this->getTheHighestDiscountPipeline($brand_code);
        $pipeline[] = $this->getTheTheLowerPricePipeline($brand_code);

        $pipeline[] = [
            '$match' => (object)$filters
        ];

        if ($sort && $sort !== ['random' => true]) {
            // `isInvenduto` is a field that states if vehicle belongs to certain dealer or is available from
            // central stock. "Invenduto" offers should always be placed after "in place" offers, and hence the
            // need of additional field to sort by.
            if (array_key_exists('isInvenduto', (array)$sort)) {
                $pipeline[] = [
                    '$addFields' => [
                        'isInvenduto' => [
                            '$cond' => [
                                'if' => '$dealer',
                                'then' => false,
                                'else' => true,
                            ]
                        ]
                    ]
                ];
            }


            // Hello again. The `distance` sort is tricky. Giving it as a parameter will make additional field
            // called `distance` that will contain simple calculated (as on the plain surface, not Earth-alike)
            // distance from the point given in filters, under `dealer.coordinates`.
            if (array_key_exists('distance', (array)$sort)) {
                // Find if there is coordinates filter available:
                if ($distance_base = $this->getDistanceBaseCoordinates($filters)) {
                    $pipeline[] = [
                        '$addFields' => [
                            'distance' => [
                                '$sqrt' => [
                                    '$add' => [
                                        [
                                            '$pow' => [
                                                [
                                                    '$subtract' => [
                                                        $distance_base['latitude'],
                                                        ['$arrayElemAt' => ['$dealer.coordinates.coordinates', 1]]
                                                    ]
                                                ],
                                                2
                                            ]
                                        ],
                                        [
                                            '$pow' => [
                                                [
                                                    '$subtract' => [
                                                        $distance_base['longitude'],
                                                        ['$arrayElemAt' => ['$dealer.coordinates.coordinates', 0]]
                                                    ]
                                                ],
                                                2
                                            ]
                                        ],
                                    ],
                                ],
                            ]
                        ]
                    ];
//                        $sort['distance'] = \FCA\StockApi\Collection\Sort\Sort::ASC;
                } else {
                    unset($sort['distance']);
                }
            }

            if (array_key_exists('popular', (array)$sort)) {
                if (is_array($sort['popular']) and count($sort['popular']) >= 1) {
                    $branches = [];

                    foreach ($sort['popular'] as $item) {
                        $branches[] = [
                            'case' => [
                                '$eq' => ['$vehicle_id', (int)$item['id']]
                            ],
                            'then' => (int)$item['visit_count']
                        ];
                    }

                    $pipeline[] = [
                        '$addFields' => [
                            'popular' => [
                                '$switch' => [
                                    'branches' => $branches,
                                    'default' => 0
                                ]
                            ]
                        ]
                    ];

                    $sort['popular'] = Sort::DESC;
                } else {
                    unset($sort['popular']);
                }
            }

            if (!Validator::isValidSortArray($sort)) {
                throw new ApiException('Sort array is invalid!');
            }

            $pipeline[] = [
                '$sort' => $sort
            ];
        }

        return $pipeline;
    }

    /**
     * @param array $filters
     * @param array|null $sort
     * @param int|null $skip
     * @return \FCA\StockApi\Document\Offer|null
     * @throws ApiException
     */
    public function findOne(array $filters = [], ?array $sort = null, ?int $skip = null): ?\FCA\StockApi\Document\Offer
    {
        $offers = $this->find($filters, $sort, 1, $skip);

        if (count($offers) > 0) {
            return $offers[0];
        }

        return null;
    }

    /**
     * @param string $field
     * @param array $filters
     * @return DistinctValue[]
     * @throws ApiException
     */
    public function findDistinct(string $field, array $filters = []): array
    {
        Validator::validateFilter($filters);

        $pipeline = $this->getBasicPipelines($filters);
        $pipeline[] = [
            '$unwind' => '$' . $field
        ];
        $pipeline[] = [
                '$group' => [
                    '_id' => '$' . $field,
                    'count' => [
                        '$sum' => 1
                    ]
                ]
        ];

        $differentValues = $this->offerCollection->aggregate($pipeline);

        /**
         * @var $distinctValues DistinctValue[]
         */
        $distinctValues = [];

        foreach ($differentValues as $doc) {
            $distinctValues[] = new DistinctValue($doc['_id'], $doc['count']);
        }

        return $distinctValues;
    }

    public function count(array $filters = []): int
    {
        Validator::validateFilter($filters);

        // Deprecated due to "$near is not allowed inside of a $match aggregation expression" error
//        return $this->offerCollection->countDocuments($filters);

        $pipeline = $this->getBasicPipelines($filters, null);

        $pipeline[] = [
            '$count' => 'the_count'
        ];

        $count = 0;

        /** @var \MongoDB\Driver\Cursor $result */
        foreach ($this->offerCollection->aggregate($pipeline) as $record) {
            /** @var \MongoDB\Model\BSONDocument $record */
            $count = (int) $record->the_count;
        }


        return $count;
    }

    /**
     * Tricky helper to extract from used filters geo-coordinates of point that is being used as a base for search (like
     * 'Bytom' or 'Rawa Mazowiecka').
     *
     * @param array $filters
     * @return array|null
     */
    private function getDistanceBaseCoordinates(array $filters = []): ?array
    {
        foreach ($filters as $operator => $filter_set) {
            foreach ($filter_set as $filter) {
                foreach ($filter as $field => $conditions) {
                    if ($field === Fields::DEALER_COORDINATES) {
                        // `$geoWithin` is just one (however the only currently used) of many ways to limit records
                        // based on the geo-region (here: sphere). There may be others used in the future, like `$near`
                        // or `$geoNear`, and they will have other structure that should be then added here.
                        if ($conditions['$geoWithin']['$centerSphere'][0] ?? false) {
                            $coordinates = $conditions['$geoWithin']['$centerSphere'][0];
                            if (isset($coordinates[0], $coordinates[1])) {
                                return [
                                    'longitude' => $conditions['$geoWithin']['$centerSphere'][0][0],
                                    'latitude' => $conditions['$geoWithin']['$centerSphere'][0][1],
                                ];
                            }
                        }
                    }
                }
            }
        }

        return null;
    }

    /**
     * @param array $filters
     * @return string|null
     */
    private function getBrandCodeFromFiltersArray(array $filters): ?string
    {
        if (!is_array($filters)) {
            return null;
        }

        foreach ($filters as $key => $value) {
            if ($key !== 'brand.code') {
                $code = $this->getBrandCodeFromFiltersArray($filters[$key]);
                if ($code !== null) {
                    return $code;
                }
            } else {
                if (is_string($value)) {
                    return $value;
                } else {
                    return $value['$in'][0];
                }
            }
        }

        return null;
    }

    private function getModelHighestDiscountByBrand(string $brand): array
    {
        /**
         * @var $docs \MongoDB\Driver\Cursor
         */
        $docs = $this->offerCollection->aggregate(
            [
                [
                    '$match' => [
                        'brand.code' => $brand
                    ]
                ],
                [
                    '$group' => [
                        '_id' => '$model.group.name',
                        'max_discount' => [
                            '$max' => [
                                '$divide' => [
                                    ['$subtract' => ['$price.base.netto', '$price.final.netto']], '$price.base.netto'
                                ]
                            ]
                        ]
                    ]
                ]
            ],
            [
                'collation' => ['locale' => 'pl'],
            ]
        );

        $result = [];

        foreach ($docs as $doc) {
            $result[] = [
                'model' => $doc['_id'],
                'max_discount' => (int)ceil($doc['max_discount'] * 10000)
            ];
        }

        return $result;
    }

    private function getTheHighestDiscountPipeline(string $brand): array
    {
        $maxModelDiscounts = $this->getModelHighestDiscountByBrand($brand);
        $branches = [];

        foreach ($maxModelDiscounts as $item) {
            $branches[] = [
                'case' => [
                    '$eq' => ['$model.group.name', $item['model']]
                ],
                'then' => (int)$item['max_discount'],
            ];
        }

        return [
            '$addFields' => [
                'labels' => [

                    '$cond' => [
                        'if' => [
                            '$eq' => [
                                ['$switch' => [
                                    'branches' => $branches,
                                    'default' => '0'
                                ]],
                                ['$ceil' => [
                                    '$multiply' => [
                                        [
                                            '$divide' => [
                                                [
                                                    '$subtract' => [
                                                        '$price.base.netto',
                                                        '$price.final.netto'
                                                    ]
                                                ],
                                                '$price.base.netto'
                                            ]
                                        ],
                                        10000
                                    ]]],
                            ]
                        ],
                        'then' => [
                            '$concatArrays' => [
                                '$labels',
                                ["Największy rabat"]
                            ]
                        ],
                        'else' => '$labels'
                    ]
                ]
            ]

        ];
    }

    private function getModelLowestPricesByBrand(string $brand): array
    {
        /**
         * @var $docs \MongoDB\Driver\Cursor
         */
        $docs = $this->offerCollection->aggregate(
            [
                [
                    '$match' => [
                        'brand.code' => $brand
                    ]
                ],
                [
                    '$group' => [
                        '_id' => '$model.group.name',
                        'lowest_price' => [
                            '$min' => '$price.final.netto'
                        ]
                    ]
                ]
            ],
            [
                'collation' => ['locale' => 'pl'],
            ]
        );

        $result = [];

        foreach ($docs as $doc) {
            $result[] = [
                'model' => $doc['_id'],
                'lowest_price' => $doc['lowest_price']
            ];
        }

        return $result;
    }

    private function getTheTheLowerPricePipeline(string $brand): array
    {
        $maxModelDiscounts = $this->getModelLowestPricesByBrand($brand);
        $branches = [];

        foreach ($maxModelDiscounts as $item) {
            $branches[] = [
                'case' => [
                    '$eq' => ['$model.group.name', $item['model']]
                ],
                'then' => (float)$item['lowest_price'],
            ];
        }

        return [
            '$addFields' => [
                'labels' => [

                    '$cond' => [
                        'if' => [
                            '$eq' => [
                                ['$switch' => [
                                    'branches' => $branches,
                                    'default' => '0'
                                ]],
                                '$price.final.netto'
                            ]
                        ],
                        'then' => [
                            '$concatArrays' => [
                                '$labels',
                                ["Najniższa cena"]
                            ]
                        ],
                        'else' => '$labels'
                    ]
                ]
            ]

        ];
    }
}
