<?php
declare(strict_types=1);

namespace FCA\StockApi\Document\Offer;

use FCA\StockApi\Document\Base;
use FCA\StockApi\Document\Offer\Price\Installment;

class Price extends Base
{
    /**
     * @var Price\Details
     */
    protected $base;

    /**
     * @var Price\Details
     */
    protected $final;

    /**
     * @var float
     */
    protected $building;

    /**
     * @var array
     */
    protected $discount;

    /**
     * @var Installment|null
     */
    protected $leasing;

    /**
     * @var float Holds netto value for LCV and brutto for other brands
     */
    protected $shipping_fee;

    /**
     * @var int|null Holds netto value for LCV and brutto for other brands
     */
    protected $omnibus_price;

    public function __construct(
        float $base_netto,
        float $base_brutto,
        float $final_netto,
        float $final_brutto,
        float $building,
        array $discount,
        array $leasing,
        float $shipping_fee,
        ?int $omnibus_price
    ) {
        $this->base = new Price\Details($base_netto, $base_brutto);
        $this->final = new Price\Details($final_netto, $final_brutto);
        $this->building = $building;
        $this->discount = $discount;
        if ($leasing) {
            $this->leasing = new Price\Installment($leasing);
        }
        $this->shipping_fee = $shipping_fee;
        $this->omnibus_price = $omnibus_price;
    }

    public function getBase(): Price\Details
    {
        return $this->base;
    }

    public function getFinal(): Price\Details
    {
        return $this->final;
    }

    public function getBuilding(): float
    {
        return $this->building;
    }

    public function getDiscount(): array
    {
        return $this->discount;
    }

    public function getLeasing(): ?Installment
    {
        return $this->leasing;
    }

    /**
     * @return float Netto value for LCV and brutto for other brands
     */
    public function getShippingFee(): float
    {
        return $this->shipping_fee;
    }
    /**
     * @return int|null Netto value for LCV and brutto for other brands
     */
    public function getOmnibusPrice(): ?int
    {
        return $this->omnibus_price;
    }

    /**
     * @return Price\Details[]
     */
    public function toArray(): array
    {
        return [
            'base' => $this->getBase()->toArray(),
            'final' => $this->getFinal()->toArray(),
            'building' => $this->getBuilding(),
            'discount' => $this->getDiscount(),
            'leasing' => $this->getLeasing() instanceof Price\Installment ? $this->getLeasing()->toArray() : null,
            'shipping_fee' => $this->getShippingFee(),
            'omnibus_price' => $this->getOmnibusPrice(),
        ];
    }
}
