<?php
declare(strict_types=1);

namespace FCA\StockApi\Document\Offer;

use FCA\StockApi\Document\Base;
use FCA\StockApi\Document\Offer\Dealer\Coordinates;

class Dealer extends Base
{
    /**
     * @var string
     */
    protected $code;

    /**
     * @var string
     */
    protected $sitecode;

    /**
     * @var string
     */
    protected $name;

    /**
     * @var string
     */
    protected $email;

    /**
     * @var string
     */
    protected $street;

    /**
     * @var string
     */
    protected $postCode;

    /**
     * @var string
     */
    protected $city;

    /**
     * @var string
     */
    protected $region;

    /**
     * @var Coordinates
     */
    protected $coordinates;

    /**
     * @var string
     */
    protected $website;

    /**
     * @var string[]
     */
    protected $phones;

    /**
     * @var bool
     */
    protected $subdealer;

    /**
     * @var array|null
     */
    protected $communicators = [];

    public function __construct(array $dealer)
    {
        $requiredKeys = [
            'code',
            'sitecode',
            'name',
            'email',
            'street',
            'post_code',
            'city',
            'region',
            'coordinates',
            'website',
            'phones',
            'subdealer',
        ];
        $this->checkKeys($requiredKeys, $dealer);

        $this->code = $dealer['code'];
        $this->sitecode = $dealer['sitecode'];
        $this->name = $dealer['name'];
        $this->email = $dealer['email'];
        $this->street = $dealer['street'];
        $this->postCode = $dealer['post_code'];
        $this->city = $dealer['city'];
        $this->region = $dealer['region'];
        $this->website = $dealer['website'];
        $this->phones = (array)$dealer['phones'];
        $this->subdealer = $dealer['subdealer'];
        $this->communicators = (array) ($dealer['communicators'] ?? []);

        $coordinates = (array)$dealer['coordinates'];
        $this->coordinates = new Coordinates(
            (float)$coordinates['coordinates'][0],
            (float)$coordinates['coordinates'][1]
        );
    }

    public function getCode(): string
    {
        return $this->code;
    }

    public function getSitecode(): string
    {
        return $this->sitecode;
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function getEmail(): string
    {
        return $this->email;
    }

    public function getStreet(): string
    {
        return $this->street;
    }

    public function getPostCode(): string
    {
        return $this->postCode;
    }

    public function getCity(): string
    {
        return $this->city;
    }

    public function getRegion(): string
    {
        return $this->region;
    }

    public function getCoordinates(): Coordinates
    {
        return $this->coordinates;
    }

    public function getWebsite(): string
    {
        return $this->website;
    }

    /**
     * @return string[]
     */
    public function getPhones(): array
    {
        return $this->phones;
    }

    /**
     * @return bool
     */
    public function isSubdealer(): bool
    {
        return $this->subdealer;
    }

    /**
     * @return array
     */
    public function getCommunicators(): array
    {
        return $this->communicators;
    }

    public function toArray(): array
    {
        return [
            'code' => $this->getCode(),
            'sitecode' => $this->getSitecode(),
            'name' => $this->getName(),
            'email' => $this->getEmail(),
            'street' => $this->getStreet(),
            'postCode' => $this->getPostCode(),
            'city' => $this->getCity(),
            'region' => $this->getRegion(),
            'coordinates' => $this->getCoordinates()->toArray(),
            'website' => $this->getWebsite(),
            'phones' => $this->getPhones(),
            'subdealer' => $this->isSubdealer(),
            'communicators' => $this->getCommunicators(),
        ];
    }
}
