<?php
declare(strict_types=1);

namespace FCA\StockApi\Collection;

use FCA\StockApi\Api;
use FCA\StockApi\Collection\Filter\Builder\Fields;
use FCA\StockApi\Collection\Filter\Builder\Validator;
use FCA\StockApi\Collection\Sort\Sort;
use FCA\StockApi\Document\DistinctValue;
use FCA\StockApi\Exception\ApiException;

class Offer
{
    /**
     * @var \MongoDB\Collection
     */
    protected $offerCollection;

    /**
     * Offer constructor.
     * @param Api $api
     * @throws ApiException
     */
    public function __construct(Api $api)
    {
        if ($api->getCollection() === null) {
            throw new ApiException('Database collection is missing!');
        }

        $this->offerCollection = $api->getCollection();
    }

    /**
     * @param array $filters
     * @param array|null $sort
     * @param int|null $limit
     * @param int|null $skip
     * @return \FCA\StockApi\Document\Offer[]
     * @throws ApiException
     */
    public function find(array $filters = [], ?array $sort = null, ?int $limit = null, ?int $skip = null): array
    {
        Validator::validateFilter($filters);

        $options = [
            'collation' => ['locale' => 'pl'],
        ];

        $pipeline = $this->getBasicPipelines($filters, $sort);

        if ($sort === ['random' => true]) {
            if ($limit === null) {
                throw new ApiException('Limit must be defined in random find!');
            }
            $pipeline[] = [
                '$sample' => [
                    'size' => $limit
                ]
            ];
        } else {
            if ($skip) {
                $pipeline[] = [
                    '$skip' => $skip
                ];
            }
            if ($limit) {
                $pipeline[] = [
                    '$limit' => $limit
                ];
            }
        }

        $docs = $this->offerCollection->aggregate(
            $pipeline,
            $options
        );

        $offers = [];

        foreach ($docs as $doc) {
            $offers[] = new \FCA\StockApi\Document\Offer((array)$doc);
        }

        $docs = null;
        unset($docs);

        return $offers;
    }

    /**
     * @param array $filters
     * @param array|null $sort
     * @return array
     * @throws ApiException
     */
    private function getBasicPipelines(array $filters = [], ?array $sort = null): array
    {
        $pipeline = [];

        // Sortowanie i filtrowanie po racie abonamentu b2b (leasingu l101, jeżeli oferta nie ma b2b)
        // Sortowanie i filtrowanie po racie abonamentu b2c (pożyczki p0p, jeżeli oferta nie ma b2c)
        $pipeline[] = [
            '$addFields' => [
                'installment-b2b-l101' => [
                    '$ifNull' => [
                        '$price.financing_info.b2b.installment',
                        '$price.financing_info.l101.installment'
                    ],
                ],
                'installment-b2c-p0p' => [
                    '$ifNull' => [
                        '$price.financing_info.b2c.installment',
                        '$price.financing_info.p0p.installment'
                    ],
                ],
            ]
        ];

        $pipeline[] = [
            '$match' => (object)$filters
        ];

        if ($sort && $sort !== ['random' => true]) {
            // `isInvenduto` is a field that states if vehicle belongs to certain dealer or is available from
            // central stock. "Invenduto" offers should always be placed after "in place" offers, and hence the
            // need of additional field to sort by.
            if (array_key_exists('isInvenduto', (array)$sort)) {
                $pipeline[] = [
                    '$addFields' => [
                        'isInvenduto' => [
                            '$cond' => [
                                'if' => '$dealer',
                                'then' => false,
                                'else' => true,
                            ]
                        ]
                    ]
                ];
            }


            // Hello again. The `distance` sort is tricky. Giving it as a parameter will make additional field
            // called `distance` that will contain simple calculated (as on the plain surface, not Earth-alike)
            // distance from the point given in filters, under `dealer.coordinates`.
            if (array_key_exists('distance', (array)$sort)) {
                // Find if there is coordinates filter available:
                if ($distance_base = $this->getDistanceBaseCoordinates($filters)) {
                    $pipeline[] = [
                        '$addFields' => [
                            'distance' => [
                                '$sqrt' => [
                                    '$add' => [
                                        [
                                            '$pow' => [
                                                [
                                                    '$subtract' => [
                                                        $distance_base['latitude'],
                                                        ['$arrayElemAt' => ['$dealer.coordinates.coordinates', 1]]
                                                    ]
                                                ],
                                                2
                                            ]
                                        ],
                                        [
                                            '$pow' => [
                                                [
                                                    '$subtract' => [
                                                        $distance_base['longitude'],
                                                        ['$arrayElemAt' => ['$dealer.coordinates.coordinates', 0]]
                                                    ]
                                                ],
                                                2
                                            ]
                                        ],
                                    ],
                                ],
                            ]
                        ]
                    ];
//                        $sort['distance'] = \FCA\StockApi\Collection\Sort\Sort::ASC;
                } else {
                    unset($sort['distance']);
                }
            }

            if (array_key_exists('first_production_year', (array)$sort)) {
                $pipeline[] = [
                    '$addFields' => [
                        'first_production_year' => [
                            '$cond' => [
                                'if' => [
                                    '$eq' => [
                                        '$year_of_production',
                                        $sort['first_production_year']
                                    ]
                                ],
                                'then' => true,
                                'else' => false,
                            ]
                        ]
                    ]
                ];
                $sort['first_production_year'] = Sort::DESC;
            }

            if (array_key_exists('first_with_label', (array)$sort)) {
                $pipeline[] = [
                    '$addFields' => [
                        'first_with_label' => [
                            '$cond' => [
                                'if' => [
                                    '$gte' => [
                                        [
                                            '$indexOfArray' => [
                                                '$labels',  $sort['first_with_label']
                                            ]
                                        ],
                                        0
                                    ]
                                ],
                                'then' => true,
                                'else' => false,
                            ]
                        ],
                    ]
                ];
                $sort['first_with_label'] = Sort::DESC;
            }

            if (array_key_exists('discount', (array)$sort)) {
                switch ($sort['discount']) {
                    case 'percent-asc':
                        $sort['discount'] = Sort::ASC;
                        $sort = $this->changeArrayKey($sort, 'discount', Fields::DISCOUNT_PERCENT);
                        break;
                    case 'percent-desc':
                        $sort['discount'] = Sort::DESC;
                        $sort = $this->changeArrayKey($sort, 'discount', Fields::DISCOUNT_PERCENT);
                        break;
                    case 'amount-asc':
                        $sort['discount'] = Sort::ASC;
                        $sort = $this->changeArrayKey($sort, 'discount', Fields::DISCOUNT_AMOUNT_BRUTTO);
                        break;
                    case 'amount-desc':
                        $sort['discount'] = Sort::DESC;
                        $sort = $this->changeArrayKey($sort, 'discount', Fields::DISCOUNT_AMOUNT_BRUTTO);
                        break;
                }
            }

            if (!Validator::isValidSortArray($sort)) {
                throw new ApiException('Sort array is invalid!');
            }

            $pipeline[] = [
                '$sort' => $sort
            ];
        }

        return $pipeline;
    }

    private function changeArrayKey(array $array, string $old_key, string $new_key)
    {
        if (!array_key_exists($old_key, $array)) {
            return $array;
        }

        $key_pos = array_search($old_key, array_keys($array));
        $arr_before = array_slice($array, 0, $key_pos);
        $arr_after = array_slice($array, $key_pos + 1);
        $arr_renamed = array($new_key => $array[$old_key]);

        return $arr_before + $arr_renamed + $arr_after;
    }

    /**
     * @param array $filters
     * @param array|null $sort
     * @param int|null $skip
     * @return \FCA\StockApi\Document\Offer|null
     * @throws ApiException
     */
    public function findOne(array $filters = [], ?array $sort = null, ?int $skip = null): ?\FCA\StockApi\Document\Offer
    {
        $offers = $this->find($filters, $sort, 1, $skip);

        if (count($offers) > 0) {
            return $offers[0];
        }

        return null;
    }

    /**
     * @param string $field
     * @param array $filters
     * @return DistinctValue[]
     * @throws ApiException
     */
    public function findDistinct(string $field, array $filters = []): array
    {
        Validator::validateFilter($filters);

        $pipeline = $this->getBasicPipelines($filters);
        $pipeline[] = [
            '$unwind' => '$' . $field
        ];
        $pipeline[] = [
            '$group' => [
                '_id' => '$' . $field,
                'count' => [
                    '$sum' => 1
                ]
            ]
        ];

        $differentValues = $this->offerCollection->aggregate($pipeline);

        /**
         * @var $distinctValues DistinctValue[]
         */
        $distinctValues = [];

        foreach ($differentValues as $doc) {
            $distinctValues[] = new DistinctValue($doc['_id'], $doc['count']);
        }

        return $distinctValues;
    }

    /**
     * @param array $filters
     * @return int
     * @throws ApiException
     */
    public function count(array $filters = []): int
    {
        Validator::validateFilter($filters);

        // Deprecated due to "$near is not allowed inside of a $match aggregation expression" error
//        return $this->offerCollection->countDocuments($filters);

        $pipeline = $this->getBasicPipelines($filters, null);

        $pipeline[] = [
            '$count' => 'the_count'
        ];

        $count = 0;

        /** @var \MongoDB\Driver\Cursor $result */
        foreach ($this->offerCollection->aggregate($pipeline) as $record) {
            /** @var \MongoDB\Model\BSONDocument $record */
            $count = (int) $record->the_count;
        }


        return $count;
    }

    /**
     * Tricky helper to extract from used filters geo-coordinates of point that is being used as a base for search (like
     * 'Bytom' or 'Rawa Mazowiecka').
     *
     * @param array $filters
     * @return array|null
     */
    private function getDistanceBaseCoordinates(array $filters = []): ?array
    {
        foreach ($filters as $operator => $filter_set) {
            foreach ($filter_set as $filter) {
                foreach ($filter as $field => $conditions) {
                    if ($field === Fields::DEALER_COORDINATES) {
                        // `$geoWithin` is just one (however the only currently used) of many ways to limit records
                        // based on the geo-region (here: sphere). There may be others used in the future, like `$near`
                        // or `$geoNear`, and they will have other structure that should be then added here.
                        if ($conditions['$geoWithin']['$centerSphere'][0] ?? false) {
                            $coordinates = $conditions['$geoWithin']['$centerSphere'][0];
                            if (isset($coordinates[0], $coordinates[1])) {
                                return [
                                    'longitude' => $conditions['$geoWithin']['$centerSphere'][0][0],
                                    'latitude' => $conditions['$geoWithin']['$centerSphere'][0][1],
                                ];
                            }
                        }
                    }
                }
            }
        }

        return null;
    }

    /**
     * @param string|null $brand
     * @return array
     */
    public function getModelHighestDiscountByBrand(?string $brand): array
    {
        $pipelines = [];

        if ($brand !== null) {
            $pipelines[] = [
                '$match' => [
                    'brand.code' => $brand
                ]
            ];
        };

        $pipelines[] = [
            '$group' => [
                '_id' => '$model.group.name',
                'max_discount' => [
                    '$max' => '$price.discount.percent'
                ]
            ]
        ];

        /**
         * @var $docs \MongoDB\Driver\Cursor
         */
        $docs = $this->offerCollection->aggregate(
            $pipelines,
            [
                'collation' => ['locale' => 'pl'],
            ]
        );

        $result = [];

        foreach ($docs as $doc) {
            $result[] = [
                'model' => $doc['_id'],
                'max_discount' => $doc['max_discount']
            ];
        }

        return $result;
    }

    /**
     * @param string|null $brand
     * @return array
     */
    public function getModelLowestPricesByBrand(?string $brand): array
    {
        $pipelines = [];

        if ($brand !== null) {
            $pipelines[] = [
                '$match' => [
                    'brand.code' => $brand
                ]
            ];
        };

        $pipelines[] = [
            '$group' => [
                '_id' => '$model.group.name',
                'lowest_price' => [
                    '$min' => '$price.final.netto'
                ]
            ]
        ];

        /**
         * @var $docs \MongoDB\Driver\Cursor
         */
        $docs = $this->offerCollection->aggregate(
            $pipelines,
            [
                'collation' => ['locale' => 'pl'],
            ]
        );

        $result = [];

        foreach ($docs as $doc) {
            $result[] = [
                'model' => $doc['_id'],
                'lowest_price' => $doc['lowest_price']
            ];
        }

        return $result;
    }
}
