<?php

namespace FCA\StockExporter;

use FCA\StockExporter\Exception\IncorrectFileType;
use FCA\StockExporter\ExportFile\CsvType;
use FCA\StockExporter\ExportFile\XlsxType;
use FCA\StockExporter\ExportFile\XmlType;

class ExportCreator
{
    /** @var \FCA\StockExporter\ExportField */
    private $export_fields;

    /** @var \FCA\StockExporter\ExportFilter */
    private $export_filter;

    /** @var \FCA\StockExporter\ExportFile\TypeInterface|null */
    private $export_object;

    /** @var string */
    private $temp_dir;

    private $available_formats = [
        CsvType::class  => CsvType::TYPE_CSV,
        XlsxType::class => XlsxType::TYPE_XLSX,
        XmlType::class  => XmlType::TYPE_XML,
    ];

    public function __construct(?string $temp_dir = null)
    {
        if (is_null($temp_dir)) {
            $this->temp_dir = sys_get_temp_dir();
        } else {
            if (!is_writable($temp_dir)) {
                throw new \RuntimeException('Given temporary directory (' . $temp_dir . ') is not writeable');
            }
            $this->temp_dir = $temp_dir;
        }
        $this->export_fields = new ExportField();
        $this->export_filter = new ExportFilter();
    }

    /**
     * @throws \FCA\StockExporter\Exception\IncorrectFileType
     */
    public function setFileType(string $type, string $name, bool $zipped = false): void
    {
        if (!class_exists($type)) {
            throw new IncorrectFileType("$type is not supported");
        }
        $this->export_object = new $type($this->export_fields, $this->export_filter, $this->temp_dir, $name);
        $this->export_object->setZipped($zipped);
        $this->export_object->initFile();
    }

    public function getExportFields(): ExportField
    {
        return $this->export_fields;
    }

    public function getExportFilter(): ExportFilter
    {
        return $this->export_filter;
    }

    public function getExportObject(): ?ExportFile\TypeInterface
    {
        return $this->export_object;
    }

    /**
     * @return string[]
     */
    public function getFields(): array
    {
        return $this->getExportFields()->getFieldNames();
    }

    /**
     * @return string[]
     */
    public function getAvailableFormats(): array
    {
        return $this->available_formats;
    }
}
