<?php

namespace FCA\StockExporter\ExportFile;

use Exception;
use FCA\StockExporter\ExportField;
use FCA\StockExporter\ExportFilter;
use FCA\StockExporter\VehicleInterface;
use ZipArchive;

abstract class BaseFile implements TypeInterface
{
    const ZIP_EXT = '.zip';

    const PROPERTY_TRUE = 'TAK';
    const PROPERTY_FALSE = 'NIE';

    /** @var \FCA\StockExporter\ExportField */
    private $export_field;

    /** @var \FCA\StockExporter\ExportFilter */
    private $export_filter;

    /** @var string */
    private $temp_dir;

    /** @var string */
    private $export_name;

    /** @var */
    private $zipped = false;

    /** @var string */
    protected $export_file_path;

    public function __construct(
        ExportField  $export_field,
        ExportFilter $export_filter,
        ?string      $temp_dir = null,
        ?string      $export_name = null
    ) {
        $this->export_name = $export_name;
        if ($export_name === null) {
            $this->export_name = "EXPORT_" . date('YmdHis');
        }
        $this->export_field = $export_field;
        $this->export_filter = $export_filter;
        $this->temp_dir = $temp_dir ?? __DIR__ . DIRECTORY_SEPARATOR;
    }

    public function getExportName(): string
    {
        return $this->export_name;
    }

    public function getFileExt(): string
    {
        return $this->file_ext;
    }

    public function getZipped(): bool
    {
        return $this->zipped;
    }

    public function setZipped(bool $zipped): void
    {
        $this->zipped = $zipped;
    }

    public function getExportFilePath(): string
    {
        return $this->export_file_path;
    }

    public function setExportFilePath(string $export_file_path): void
    {
        $this->export_file_path = $export_file_path;
    }

    public function getExportField(): ExportField
    {
        return $this->export_field;
    }

    public function getExportFilter(): ExportFilter
    {
        return $this->export_filter;
    }

    public function getTempDir(): ?string
    {
        return $this->temp_dir;
    }

    /**
     * @throws \Exception
     */
    public function zippingIfRequired(string $file_path): void
    {
        if ($this->getZipped()) {
            $zip_file_path = $this->getTempDir() . DIRECTORY_SEPARATOR . $this->getExportName() . self::ZIP_EXT;
            $zip = new ZipArchive();
            if ($zip->open($zip_file_path, ZipArchive::CREATE) !== true) {
                throw new Exception('Can not create ZIP file');
            }
            $zip->addFile($file_path, $this->getExportName() . $this->getFileExt());
            $zip->close();
            $this->setExportFilePath($zip_file_path);
            if (file_exists($file_path)) {
                @unlink($file_path);
            }
        }
    }

    /**
     * @param \FCA\StockExporter\VehicleInterface $vehicle
     * @param string                              $field
     * @return string
     */
    public function fieldDataFormatFormatter(VehicleInterface $vehicle, string $field): string
    {
        switch ($field) {
            case ExportField::KM0:
            case ExportField::DEMO:
            case ExportField::DEALER_ADD:
                return $vehicle->getProperty($field, false) ? self::PROPERTY_TRUE : self::PROPERTY_FALSE;
            case ExportField::MODIFIED:
                return !empty($vehicle->getProperty($field))
                    ? date('Y-m-d H:i:s', $vehicle->getProperty($field))
                    : '';
            case ExportField::BASE_PRICE:
            case ExportField::DISCOUNT_MANUFACTURER:
            case ExportField::DISCOUNT_DEALER:
            case ExportField::PRICE:
            case ExportField::LEASING_INSTALLMENT:
            case ExportField::BUILDING_PRICE:
                return !empty($vehicle->getProperty($field))
                    ? number_format((float)$vehicle->getProperty($field, 0), 2, ',', '')
                    : '0.00';
            case ExportField::EQUIPMENT_OPTIONAL:
                return (!empty($vehicle->getProperty($field)) and is_array($vehicle->getProperty($field)))
                    ? implode(',', $vehicle->getProperty($field, '[]'))
                    : '';
            case ExportField::TAGS:
                return $vehicle->getProperty($field) != '[]' ? $vehicle->getProperty($field, '') : '';
            default:
                return (string)$vehicle->getProperty($field, '');
        }
    }
}
