<?php

namespace FCA\StockExporter\ExportFile;

use FCA\StockExporter\VehicleInterface;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Exception;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\BaseWriter;

/**
 * @property \PhpOffice\PhpSpreadsheet\Spreadsheet         spreadsheet
 * @property \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet sheet
 * @method getExportName()
 */
trait SpreadsheetTrait
{
    /**
     * @throws \PhpOffice\PhpSpreadsheet\Exception
     */
    public function initSpreadsheet(): void
    {
        $this->spreadsheet = new Spreadsheet();
        $this->sheet = $this->spreadsheet->getActiveSheet();
        $this->sheet->setTitle($this->getExportName());

        $this->spreadsheet->getProperties()
            ->setCompany('Stellantis N.V.')
            ->setCreator('Unicorn Interactive')
            ->setTitle($this->getExportName())
            ->setSubject($this->getExportName())
            ->setDescription($this->getExportName());

        $this->sheet
            ->getRowDimension(1)
            ->setRowHeight(31);

        $default_font = $this->spreadsheet->getDefaultStyle()->getFont();
        $default_font->setName('Arial');
        $default_font->setSize('10');
    }

    /**
     * @param \FCA\StockExporter\VehicleInterface $vehicle
     */
    public function addVehicleToFile(VehicleInterface $vehicle): void
    {
        if ($this->getExportFilter()->checkVehicle($vehicle)) {
            $last_row = $this->sheet->getHighestRow() + 1;
            $col_index = 1;
            foreach ($this->getExportField()->getFieldKeys() as $field) {
                if (in_array($field, $this->getExportField()->getSelectedFields())) {
                    try {
                        $this->sheet->getCell($this->getCellAddress($last_row, $col_index))
                            ->setValueExplicit(
                                $this->fieldDataFormatFormatter($vehicle, $field),
                                DataType::TYPE_STRING
                            );
                    } catch (Exception $e) {
                        $this->sheet->setCellValue(
                            $this->getCellAddress($last_row, $col_index),
                            $this->fieldDataFormatFormatter($vehicle, $field)
                        );
                    }
                    unset($value);
                    $col_index++;
                }
            }
        }
    }

    /**
     * @throws \Exception
     */
    public function closeSpreadsheetFile(BaseWriter $writer): void
    {
        $file_path = $this->getTempDir() . DIRECTORY_SEPARATOR . $this->getExportName() . $this->getFileExt();
        $this->setExportFilePath($file_path);
        $writer->save($file_path);
        $this->zippingIfRequired($file_path);
    }

    /**
     * @param int $row
     * @param int $col
     * @return string
     */
    protected function getCellAddress(int $row, int $col): string
    {
        return Coordinate::stringFromColumnIndex($col) . $row;
    }
}
