<?php

namespace FCA\StockExporter\ExportFile;

use FCA\StockExporter\ExportField;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class XlsxType extends BaseFile
{
    use SpreadsheetTrait;

    const TYPE_XLSX = 'XLSX';

    /** @var string */
    protected $file_ext = '.xlsx';

    /** @var Spreadsheet */
    private $spreadsheet;

    /** @var \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet */
    private $sheet;

    private $column_sizes = [
        [
            'size' => 8,
            'fields' => [
                ExportField::ID,
                ExportField::BRAND_ID,
                ExportField::STATUS,
                ExportField::PREVIOUS_STATUS,
                ExportField::IMAGES_COUNT,
                ExportField::KM0,
                ExportField::DEMO,
                ExportField::MILEAGE,
                ExportField::COLOR_CODE,
                ExportField::INTERIOR_CODE,
            ]
        ],
        [
            'size' => 16,
            'fields' => [
                ExportField::MODEL_ID,
                ExportField::SERIAL_NO,
                ExportField::DEALER_SINCOM,
                ExportField::DEALER_ADD,
                ExportField::VERSION_ID,
                ExportField::MODIFIED,
                ExportField::PRODUCTION_YEAR,
                ExportField::PRODUCTION_DATE,
                ExportField::FRY_DATE,
                ExportField::CCF_DATE,
                ExportField::OCF_DATE,
                ExportField::ORDER_DATE,
                ExportField::LEASING_INSTALLMENT,
                ExportField::BUILDING_PRICE,
                ExportField::COLOR_CODE,
                ExportField::INTERIOR_CODE,
            ]
        ],
        [
            'size' => 24,
            'fields' => [
                ExportField::BRAND_NAME,
                ExportField::DEALER_NAME,
                ExportField::VERSION_NAME,
                ExportField::MODEL_NAME,
                ExportField::MODEL_GROUP_NAME,
                ExportField::BASE_PRICE,
                ExportField::DISCOUNT_MANUFACTURER,
                ExportField::DISCOUNT_DEALER,
                ExportField::PRICE,
                ExportField::STATUS_NAME,
                ExportField::PREVIOUS_STATUS_NAME,
                ExportField::VIN,
                ExportField::TAGS,
                ExportField::SQUARE,
                ExportField::COLOR_NAME,
                ExportField::INTERIOR_NAME,

            ]
        ],
        [
            'size' => 40,
            'fields' => [
                ExportField::EQUIPMENT_OPTIONAL,
            ]
        ]
    ];

    /**
     * Create export file object, set top row and set widths of columns
     *
     * @throws \PhpOffice\PhpSpreadsheet\Exception
     */
    public function initFile(): void
    {
        $this->initSpreadsheet();

        $col_index = 1;
        foreach ($this->getExportField()->getFieldKeys() as $field) {
            if (in_array($field, $this->getExportField()->getSelectedFields())) {
                /** Set header names */
                $this->sheet->setCellValue(
                    $this->getCellAddress(1, $col_index),
                    $this->getExportField()->getFieldName($field)
                );
                /** Set width of column */
                $this->sheet->getCell($this->getCellAddress(1, $col_index));
                $this->sheet
                    ->getColumnDimensionByColumn($col_index)
                    ->setWidth($this->getColumnSizeForField($field));

                /*** Set header styles ***/
                $style = $this->sheet->getCell($this->getCellAddress(1, $col_index))->getStyle();

                $style->getFont()->setBold(true);
                $style->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                $style->getFill()
                    ->setFillType(Fill::FILL_SOLID);

                $style->getFill()
                    ->getStartColor()
                    ->setRGB('DDDDDD');
                $col_index++;
            }
        }
    }

    /**
     * @throws \Exception
     */
    public function closeFile(): void
    {
        $writer = new Xlsx($this->spreadsheet);
        $this->closeSpreadsheetFile($writer);
    }

    private function getColumnSizeForField(string $field): int
    {
        foreach ($this->column_sizes as $size) {
            if (in_array($field, $size['fields'])) {
                return $size['size'];
            }
        }

        return 12;
    }
}
