<?php

namespace FCAPoland\VehicleApiHelper\Brand;

use FCAPoland\VehicleApiHelper\Brand;
use FCAPoland\VehicleApiHelper\Exception;
use FCAPoland\VehicleApiHelper\VehicleHelper;
use Psr\SimpleCache\CacheInterface;

/**
 * Class Collection
 */
class Collection
{
    const API_BRANDS_URL = 'https://api.fcapoland.pl/brands';

    const BRANDS_CACHE_KEY = 'brands';
    const BRANDS_CACHE_KEY_FOR_BACKUP = 'brands_backup';

    const CACHE_TTL = 3600 * 24; // 24 hour
    const CACHE_TTL_FOR_BACKUP = 3600 * 72; // 72 h

    /**
     * @var CacheInterface
     */
    private $cache;

    /** @var Collection */
    private $brands;

    /**
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function fetchBrands()
    {
        // Dwa przypadki cache jest o go nie ma. Tu rozpatrujemy gdy cache jest
        if ($this->cache != null) {
            // Próba pobrania marek z cache
            $brands = $this->cache->get($this->getCacheKey());
            // Gdy w cache nie ma marek
            if (!$brands) {
                // Próba pobrania z API i zapisu do cache i cache awaryjnego
                try {
                    $brands = $this->fetch();
                    $this->cache->set($this->getCacheKey(), $brands, self::CACHE_TTL);
                    $this->cache->set($this->getCacheKeyForBackup(), $brands, self::CACHE_TTL_FOR_BACKUP);
                } catch (Exception $e) {
                    // Gdy pobranie marek z api nie jest możliwe próba pobrania z cache awaryjnego
                    $brands = $this->cache->get($this->getCacheKeyForBackup());
                    // Gdy nie ma marek w cache awaryjnym błąd?
                    if (!$brands) {
                        throw new Exception($e->getMessage(), null, $e);
                    }
                }
            }
        } else {
            // Próba pobrania marek z api gdy pobierze zwraca treść w innym wypadku błąd
            try {
                $brands = $this->fetch();
            } catch (Exception $e) {
                throw new Exception($e->getMessage(), null, $e);
            }
        }
        if (!$this->isJson($brands)) {
            throw new Exception('Brands json has invalid format!');
        }
        $brands  = json_decode($brands, true);

        foreach ($brands as $brand) {
            $this->brands[$brand['id']] = new Brand(
                $brand['id'],
                $brand['name'],
                $brand['slug']
            );
        }
        return $this;
    }

    /**
     * @param $id
     * @return Brand
     * @throws Exception
     */
    public function getBrandById($id)
    {
        if (!isset($this->brands[$id])) {
            throw new Exception("Brand with id: '{$id}' do not found");
        }
        return $this->brands[$id];
    }

    /**
     * @param $slug
     * @return Brand
     * @throws Exception
     */
    public function getBrandBySlug($slug)
    {
        /** @var Brand $brand */
        foreach ($this->brands as $brand) {
            if ($brand->getSlug() == $slug) {
                return $brand;
            }
        }
        throw new Exception("Brand with slug: '{$slug}' do not found");
    }

    /**
     * @return Collection
     */
    public function getAllBrands()
    {
        return $this->brands;
    }

    /**
     * @return array
     */
    public function getAllBrandAsArray()
    {
        $brands = [];
        /** @var Brand $brand */
        foreach ($this->brands as $brand) {
            $brands[$brand->getId()] = $brand->asArray();
        }
        return $brands;
    }

    /**
     * @param CacheInterface $cache
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
    }

    /**
     * @return string
     */
    private function getCacheKey()
    {
        return VehicleHelper::CACHE_KEY . '-' . static::BRANDS_CACHE_KEY;
    }

    /**
     * @return string
     */
    private function getCacheKeyForBackup()
    {
        return VehicleHelper::CACHE_KEY . '-' . static::BRANDS_CACHE_KEY_FOR_BACKUP;
    }

    /**
     * @return mixed
     * @throws Exception
     */
    private function fetch()
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, static::API_BRANDS_URL);
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, 'FCA Vehicle Helper');
            $query = curl_exec($curl_handle);
            $httpcode = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $httpcode >= 400) {
                throw new Exception('Unable to fetch brands data from FCA API');
            }

            return $query;
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), null, $e);
        }
    }

    /**
     * @param $string
     * @return bool
     */
    private function isJson($string)
    {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }
}
