<?php

namespace FCAPoland\VehicleApiHelper\Model;

use FCAPoland\VehicleApiHelper\Brand;
use FCAPoland\VehicleApiHelper\Model;
use FCAPoland\VehicleApiHelper\Exception;
use FCAPoland\VehicleApiHelper\VehicleHelper;
use Psr\SimpleCache\CacheInterface;
use Psr\SimpleCache\InvalidArgumentException;

/**
 * Class Collection
 */
class Collection
{
    const API_MODELS_URL = 'https://api.fcapoland.pl/models';

    const MODELS_CACHE_KEY = 'models';
    const MODELS_CACHE_KEY_FOR_BACKUP = 'models_backup';

    const CACHE_TTL = 3600 * 24; // 24 hour
    const CACHE_TTL_FOR_BACKUP = 3600 * 72; // 72 h

    /** @var CacheInterface|null */
    private $cache;

    /** @var Collection */
    private $models;

    /** @var Brand\Collection  */
    private $brands;

    /**
     * Collection constructor.
     */
    public function __construct()
    {
        $this->brands = new Brand\Collection();
    }

    /**
     * @throws Exception
     * @throws InvalidArgumentException
     */
    public function fetchModels()
    {
        $this->brands->fetchBrands();

        // Dwa przypadki cache jest i go nie ma. Tu rozpatrujemy, gdy cache jest
        if ($this->cache != null) {
            // Próba pobrania modeli z cache
            $models = $this->cache->get($this->getCacheKey());
            // Gdy w cache nie ma modeli
            if (!$models) {
                // Próba pobrania z API i zapisu do cache i cache awaryjnego
                try {
                    $models = $this->fetch();
                    $this->cache->set($this->getCacheKey(), $models, self::CACHE_TTL);
                    $this->cache->set($this->getCacheKeyForBackup(), $models, self::CACHE_TTL_FOR_BACKUP);
                } catch (Exception $e) {
                    // Gdy pobranie modeli z api nie jest możliwe próba pobrania z cache awaryjnego
                    $models = $this->cache->get($this->getCacheKeyForBackup());
                    // Gdy nie ma modeli w cache awaryjnym błąd?
                    if (!$models) {
                        throw new Exception($e->getMessage(), null, $e);
                    }
                }
            }
        } else {
            // Próba pobrania modeli z api, gdy pobierze zwraca treść w innym wypadku błąd
            try {
                $models = $this->fetch();
            } catch (Exception $e) {
                throw new Exception($e->getMessage(), null, $e);
            }
        }
        if (!$this->isJson($models)) {
            throw new Exception('Models json has invalid format!');
        }
        $models  = json_decode($models, true);

        foreach ($models as $model) {
            $brand = $this->brands->getBrandById($model['brand']['id']);
            $this->models[] = new Model(
                $model['id'],
                $model['code'],
                $model['name'],
                $brand
            );
        }
        return $this;
    }

    /**
     * @param $brand_id
     * @param $model_id
     * @param bool $as_array
     * @return mixed
     * @throws Exception
     */
    public function getModelByIdAndBrand($brand_id, $model_id, $as_array = false)
    {
        /** @var Model $model */
        foreach ($this->models as $model) {
            if ($model->getBrand()->getId() == $brand_id && $model->getId() == $model_id) {
                return $as_array ? $model->asArray() : $model;
            }
        }
        throw new Exception("Model with id: '{$model_id}' and brand: {$brand_id} do not found");
    }

    /**
     * @return Collection
     */
    public function getAllModels()
    {
        return $this->models;
    }

    /**
     * @return array
     */
    public function getAllModelsAsArray()
    {
        $models = [];
        /** @var Model $model */
        foreach ($this->models as $model) {
            $models[] = $model->asArray();
        }
        return $models;
    }

    /**
     * @param Brand $brand
     * @param bool $as_array
     * @return array
     */
    public function getModelForBrand(Brand $brand, $as_array = false)
    {
        $models = [];
        /** @var Model $model */
        foreach ($this->models as $model) {
            if ($brand->getSlug() == $model->getBrand()->getSlug()) {
                $models[] = $as_array ? $model->asArray() : $model;
            }
        }
        return $models;
    }

    /**
     * @param CacheInterface $cache
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
        $this->brands->setCache($this->cache);
    }

    /**
     * @return string
     */
    private function getCacheKey()
    {
        return VehicleHelper::CACHE_KEY . '-' . static::MODELS_CACHE_KEY;
    }

    /**
     * @return string
     */
    private function getCacheKeyForBackup()
    {
        return VehicleHelper::CACHE_KEY . '-' . static::MODELS_CACHE_KEY_FOR_BACKUP;
    }

    /**
     * @return bool|string
     * @throws Exception
     */
    private function fetch()
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, static::API_MODELS_URL);
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, 'FCA Vehicle Helper');
            $query = curl_exec($curl_handle);
            $httpcode = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $httpcode >= 400) {
                throw new Exception('Unable to fetch brands data from FCA API');
            }

            return $query;
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), null, $e);
        }
    }

    /**
     * @param $string
     * @return bool
     */
    private function isJson($string)
    {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }
}
