<?php

declare(strict_types=1);

use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use Stellantis\ExPsaApiHelper\Dealers;
use PHPUnit\Framework\TestCase;
use Stellantis\ExPsaApiHelper\Model\Brand;
use Stellantis\ExPsaApiHelper\Model\Dealer;
use Stellantis\ExPsaApiHelper\Model\Region;
use Stellantis\ExPsaApiHelper\Filter\Dealer as DealerFilter;

class DealersTest extends TestCase
{
    private MockHandler $mock;
    private HandlerStack $handlerStack;
    private Dealers $dealers;

    protected function setUp(): void
    {
        parent::setUp();

        // Create a mock using Guzzle helpers (https://docs.guzzlephp.org/en/stable/testing.html)
        $this->mock = new MockHandler([
            new Response(200, [], file_get_contents(__DIR__ . DIRECTORY_SEPARATOR . 'dealers.json')),
        ]);
        $this->handlerStack = HandlerStack::create($this->mock);
        $this->dealers = new Dealers('', null, null, $this->handlerStack);
    }

    public function testGetNoFilter()
    {
        $result = $this->dealers->get();
        $this->assertEquals(180, $result->count());
    }

    public function testGetAllCitroen()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setBrands(Brand::NAME_CITROEN)
        );
        $this->assertEquals(54, $result->count());
    }

    public function testGetAllOpel()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setBrands(Brand::CODE_OPEL)
        );
        $this->assertEquals(77, $result->count());
    }

    public function testGetAllPeugeot()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setBrands(Brand::NAME_PEUGEOT)
        );
        $this->assertEquals(43, $result->count());
    }

    public function testGetAllDS()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setBrands(Brand::NAME_DS)
        );
        $this->assertEquals(6, $result->count());
    }

    public function testSingleDealerObject()
    {
        $result = $this->dealers->get();
        $this->assertInstanceOf(Dealer::class, $result->current());
    }

    public function testActiveFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setActive(true)
        );
        $this->assertEquals(173, $result->count());
    }

    public function testInactiveFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setActive(false)
        );
        $this->assertEquals(7, $result->count());
    }

    public function testClearActiveFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setActive(false)->clearActive()
        );
        $this->assertEquals(180, $result->count());
    }

    public function testRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI('PL6054')
        );
        $this->assertEquals(1, $result->count());
    }

    public function testMultipleRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI('PL6054', 'PLC076')
        );
        $this->assertEquals(2, $result->count());
    }

    public function testClearRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI('PL6054', 'PLC076')->clearRRDI()
        );
        $this->assertEquals(180, $result->count());
    }

    public function testAddRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI('PL6054')->addRRDI('PLC076')
        );
        $this->assertEquals(2, $result->count());
    }

    public function testNotExistingRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI('foo')
        );
        $this->assertEquals(0, $result->count());
    }

    public function testRRDI7Filter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI7('PLC076P')
        );
        $this->assertEquals(1, $result->count());
    }

    public function testMultipleRRDI7Filter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI7('PLC076P', 'PL6054P')
        );
        $this->assertEquals(2, $result->count());
    }

    public function testClearRRDI7Filter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI7('PLC076P', 'PL6054P')->clearRRDI7()
        );
        $this->assertEquals(180, $result->count());
    }

    public function testAddRRDI7Filter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI7('PLC076P')->addRRDI7('PL6054P')
        );
        $this->assertEquals(2, $result->count());
    }

    public function testNotExistingRRDI7Filter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRRDI7('foo')
        );
        $this->assertEquals(0, $result->count());
    }

    public function testParentRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setParentRRDI('PL0061')
        );
        $this->assertEquals(3, $result->count());
    }

    public function testMultipleParentRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setParentRRDI('PL0061', 'PL0014')
        );
        $this->assertEquals(6, $result->count());
    }

    public function testClearParentRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setParentRRDI('PL0061', 'PL0014')->clearParentRRDI()
        );
        $this->assertEquals(180, $result->count());
    }

    public function testAddParentRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setParentRRDI('PL0061')->addParentRRDI('PL0014')
        );
        $this->assertEquals(6, $result->count());
    }

    public function testNotExistingParentRRDIFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setParentRRDI('foo')
        );
        $this->assertEquals(0, $result->count());
    }

    public function testBACFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setBAC('00000000733')
        );
        $this->assertEquals(1, $result->count());
    }

    public function testMultipleBACFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setBAC('00000000733', '00000243143')
        );
        $this->assertEquals(2, $result->count());
    }

    public function testClearBACFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setBAC('00000000733', '00000243143')->clearBAC()
        );
        $this->assertEquals(180, $result->count());
    }

    public function testAddBACFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setBAC('00000000733')->addBAC('00000243143')
        );
        $this->assertEquals(2, $result->count());
    }

    public function testNotExistingBACFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setBAC('foo')
        );
        $this->assertEquals(0, $result->count());
    }

    public function testRegionFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())->setRegions((new Region())->setLabel('podkarpackie'))
        );
        $this->assertEquals(8, $result->count());
    }

    public function testMultipleRegionFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())
                ->setRegions(
                    (new Region())->setLabel('łódzkie'),
                    (new Region())->setLabel('lubuskie')
                )
        );
        $this->assertEquals(13, $result->count());
    }

    public function testClearRegionFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())
                ->setRegions(
                    (new Region())->setLabel('mazowieckie'),
                )
                ->clearRegions()
        );
        $this->assertEquals(180, $result->count());
    }

    public function testAddRegionFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())
                ->setRegions(
                    (new Region())->setLabel('mazowieckie'),
                )
                ->addRegions((new Region())->setValue('woj8'))
        );
        $this->assertEquals(28, $result->count());
    }

    public function testCompoundFilter()
    {
        $result = $this->dealers->get(
            (new DealerFilter())
                ->setRegions(
                    (new Region())->setLabel('mazowieckie'),
                )
                ->setBrands('Opel')
                ->setActive(false)
        );
        $this->assertEquals(3, $result->count());
    }
}
