<?php

declare(strict_types=1);

namespace Stellantis\ExPsaApiHelper;

use Stellantis\ExPsaApiHelper\Exception\ExPSAAPIException;
use Stellantis\ExPsaApiHelper\Model\Brand;
use Stellantis\ExPsaApiHelper\Model\Model;

class Models extends Base
{
    public const ENDPOINT = 'cars';
    public const CACHE_KEY = 'models';

    public function get(?Filter\Model $filter = null): \SplObjectStorage
    {
        $models = $this->fetch();
        if ($filter === null) {
            return $models;
        }

        while ($models->valid()) {
            /** @var Model $model */
            $model = $models->current();
            $models->next();

            // Filtering happens here:
            if ($brands = $filter->getBrands()) {
                if (!in_array($model->getBrand(), $brands)) {
                    $models->detach($model);
                    continue;
                }
            }
            unset($brands);

            if ($lcdv6 = $filter->getLcdv6()) {
                if (!in_array($model->getLcdv6(), $lcdv6)) {
                    $models->detach($model);
                    continue;
                }
            }
            unset($lcdv6);

            if ($types = $filter->getTypes()) {
                if (!in_array($model->getType(), $types)) {
                    $models->detach($model);
                    continue;
                }
            }
            unset($types);

            if ($model_years = $filter->getModelYears()) {
                if (!in_array($model->getModelYear(), $model_years)) {
                    $models->detach($model);
                    continue;
                }
            }
            unset($model_years);

            if ($vehicle_notes = $filter->getVehicleNotes()) {
                if (!in_array($model->getVehicleNote(), $vehicle_notes)) {
                    $models->detach($model);
                    continue;
                }
            }
            unset($vehicle_notes);
        }

        $models->rewind();

        return $models;
    }

    private function fetch(): \SplObjectStorage
    {
        $models = $this->loadFromCache(self::CACHE_KEY);
        if ($models === null) {
            $response = $this->client->get(self::ENDPOINT);
            try {
                $decoded = json_decode($response->getBody()->getContents(), true, JSON_THROW_ON_ERROR);
            } catch (\JsonException $exception) {
                throw new ExPSAAPIException('Invalid data fetched from external API: ' . $exception->getMessage());
            }
            $models = new \SplObjectStorage();
            foreach ($decoded as $brand_slug => $types) {
                $brand = new Brand();
                $brand->setCode(preg_replace('/^cars_/', '', $brand_slug));
                foreach ($types as $type => $type_models) {
                    foreach ($type_models as $model_code => $model_data) {
                        $model = (new Model())->setBrand($brand);
                        if (isset($model_data['nazwa']) and $model_data['nazwa']) {
                            $model->setName($model_data['nazwa']);
                        }
                        if (isset($model_data['lcdv6']) and $model_data['lcdv6']) {
                            $model->setLcdv6($model_data['lcdv6']);
                        }
                        if (isset($model_data['type']) and $model_data['type']) {
                            $model->setType($model_data['type']);
                        }
                        if (isset($model_data['model_year'])) {
                            $model->setModelYear((int) $model_data['model_year']);
                        }
                        if (isset($model_data['vehicle_note']) and $model_data['vehicle_note']) {
                            $model->setVehicleNote($model_data['vehicle_note']);
                        }
                        /** @psalm-suppress InvalidArgument */
                        $models->attach($model);
                        unset($model);
                    }
                    unset($model_code, $model_data);
                }
                unset($type, $type_models);
            }
            unset($brand_slug, $types);
            $this->saveToCache(self::CACHE_KEY, $models);
        }

        return $models;
    }
}
