<?php

declare(strict_types=1);

namespace Stellantis\ExPsaApiHelper;

use Stellantis\ExPsaApiHelper\Exception\ExPSAAPIException;
use Stellantis\ExPsaApiHelper\Model\Address;
use Stellantis\ExPsaApiHelper\Model\Brand;
use Stellantis\ExPsaApiHelper\Model\Coordinate;
use Stellantis\ExPsaApiHelper\Model\Dealer;
use Stellantis\ExPsaApiHelper\Model\Region;

class Dealers extends Base
{
    public const ENDPOINT = 'dealers';
    public const CACHE_KEY = 'dealers';

    public function get(?Filter\Dealer $filter = null): \SplObjectStorage
    {
        $dealers = $this->fetch();
        if ($filter === null) {
            return $dealers;
        }

        while ($dealers->valid()) {
            /** @var Dealer $dealer */
            $dealer = $dealers->current();
            $dealers->next();

            // Filtering happens here:
            if ($filter->getActive() === true) {
                if (!$dealer->isActive()) {
                    $dealers->detach($dealer);
                    continue;
                }
            } elseif ($filter->getActive() === false) {
                if ($dealer->isActive()) {
                    $dealers->detach($dealer);
                    continue;
                }
            }

            if ($brands = $filter->getBrands()) {
                if (!in_array($dealer->getBrand(), $brands)) {
                    $dealers->detach($dealer);
                    continue;
                }
            }
            unset($brands);

            if ($rrdi = $filter->getRRDI()) {
                if (!in_array($dealer->getRRDI(), $rrdi)) {
                    $dealers->detach($dealer);
                    continue;
                }
            }
            unset($rrdi);

            if ($rrdi7 = $filter->getRRDI7()) {
                if (!in_array($dealer->getRRDI7(), $rrdi7)) {
                    $dealers->detach($dealer);
                    continue;
                }
            }
            unset($rrdi7);

            if ($parent_rrdi = $filter->getParentRRDI()) {
                if (!in_array($dealer->getParentRRDI(), $parent_rrdi)) {
                    $dealers->detach($dealer);
                    continue;
                }
            }
            unset($parent_rrdi);

            if ($bac = $filter->getBAC()) {
                if (!in_array($dealer->getBAC(), $bac)) {
                    $dealers->detach($dealer);
                    continue;
                }
            }
            unset($bac);

            if ($regions = $filter->getRegions()) {
                if (!in_array($dealer->getRegion(), $regions)) {
                    $dealers->detach($dealer);
                    continue;
                }
            }
            unset($regions);
        }

        $dealers->rewind();

        return $dealers;
    }

    private function fetch(): \SplObjectStorage
    {
        $dealers = $this->loadFromCache(self::CACHE_KEY);
        if ($dealers === null) {
            $response = $this->client->get(self::ENDPOINT);
            try {
                $decoded = json_decode($response->getBody()->getContents(), true, JSON_THROW_ON_ERROR);
            } catch (\JsonException $exception) {
                throw new ExPSAAPIException('Invalid data fetched from external API: ' . $exception->getMessage());
            }
            $dealers = new \SplObjectStorage();
            foreach ($decoded as $brand_slug => $brand_dealers) {
                $brand = new Brand();
                $brand->setCode(preg_replace('/^dealers_/', '', $brand_slug));
                foreach ($brand_dealers as $code => $dealer_data) {
                    $dealer = (new Dealer())->setBrand($brand);
                    if (isset($dealer_data['nazwa']) and $dealer_data['nazwa']) {
                        $dealer->setName(trim($dealer_data['nazwa']));
                    }
                    if (isset($dealer_data['is_active']) and $dealer_data['is_active']) {
                        $dealer->setActive(true);
                    }
                    if (isset($dealer_data['rrdi']) and $dealer_data['rrdi']) {
                        $dealer->setRRDI(trim($dealer_data['rrdi']));
                    }
                    if (isset($dealer_data['rrdi7']) and $dealer_data['rrdi7']) {
                        $dealer->setRRDI7(trim($dealer_data['rrdi7']));
                    }
                    if (isset($dealer_data['parent_rrdi']) and $dealer_data['parent_rrdi']) {
                        $dealer->setParentRRDI(trim($dealer_data['parent_rrdi']));
                    }
                    if (isset($dealer_data['bac']) and $dealer_data['bac']) {
                        $dealer->setBAC(trim($dealer_data['bac']));
                    }
                    $address = new Address();
                    if (isset($dealer_data['miasto']) and $dealer_data['miasto']) {
                        $address->setCity(trim($dealer_data['miasto']));
                    }
                    if (isset($dealer_data['adres_ulica']) and $dealer_data['adres_ulica']) {
                        $address->setStreet(trim($dealer_data['adres_ulica']));
                    }
                    if (isset($dealer_data['adres_numer']) and $dealer_data['adres_numer']) {
                        $address->setBuildingNo(trim($dealer_data['adres_numer']));
                    }
                    // Following non-strict comparison is purposeful to check against empty object (no city, no street,
                    // no building number). Strict comparison would always return `false`.
                    if ($address != new Address()) {
                        $dealer->setAddress($address);
                    }
                    if (
                        isset($dealer_data['wojewodztwo'])
                        and $dealer_data['wojewodztwo']
                        and is_array($dealer_data['wojewodztwo'])
                    ) {
                        $region = new Region();
                        if (isset($dealer_data['wojewodztwo']['label']) and $dealer_data['wojewodztwo']['label']) {
                            $region->setLabel(trim($dealer_data['wojewodztwo']['label']));
                        }
                        if (isset($dealer_data['wojewodztwo']['value']) and $dealer_data['wojewodztwo']['value']) {
                            $region->setValue(trim($dealer_data['wojewodztwo']['value']));
                        }
                        // Following non-strict comparison is purposeful to check against empty object (no label,
                        // no value). Strict comparison would always return `false`.
                        if ($region != new Region()) {
                            $dealer->setRegion($region);
                        }
                    }
                    if (isset($dealer_data['telefon']) and $dealer_data['telefon']) {
                        $dealer->setPhoneNo(trim($dealer_data['telefon']));
                    }
                    if (isset($dealer_data['mail']) and $dealer_data['mail']) {
                        $dealer->setEmail(trim($dealer_data['mail']));
                    }
                    if (isset($dealer_data['mapa_lat']) and $dealer_data['mapa_lat']) {
                        $mapa_lat = trim(preg_replace('/[^0-9.]/', '', $dealer_data['mapa_lat']), ' 0');
                        $latitude = floatval($mapa_lat);
                        if ((string)$latitude !== $mapa_lat) {
                            throw new ExPSAAPIException(
                                'Could not properly parse latitude: ' . $mapa_lat . ' != ' . $latitude
                            );
                        }
                    }
                    if (isset($dealer_data['mapa_lng']) and $dealer_data['mapa_lng']) {
                        $mapa_lng = trim(preg_replace('/[^0-9.]/', '', $dealer_data['mapa_lng']), ' 0');
                        $longitude = floatval($mapa_lng);
                        if ((string)$longitude !== $mapa_lng) {
                            throw new ExPSAAPIException(
                                'Could not properly parse longitude: ' . $mapa_lng . ' != ' . $longitude
                            );
                        }
                    }
                    if (isset($latitude, $longitude)) {
                        $dealer->setCoordinate(new Coordinate($latitude, $longitude));
                    }

                    /** @psalm-suppress InvalidArgument */
                    $dealers->attach($dealer);
                    unset($dealer);
                }
                unset($code, $dealer_data);
            }
            unset($brand_slug, $brand_dealers);
            $this->saveToCache(self::CACHE_KEY, $dealers);
        }

        return $dealers;
    }
}
