<?php

declare(strict_types=1);

namespace Stellantis\ExPsaApiHelper\Filter\Traits;

use Stellantis\ExPsaApiHelper\Exception\ExPSAAPIException;
use Stellantis\ExPsaApiHelper\Model\Brand;

trait BrandsTrait
{
    private array $brands = [];

    /**
     * @return Brand[]
     */
    public function getBrands(): array
    {
        return $this->brands;
    }

    /**
     * @param array $brands Brand codes, names or objects
     * @return $this
     * @throws ExPSAAPIException
     */
    public function setBrands(...$brands): self
    {
        foreach ($brands as &$brand) {
            /** @var mixed $brand */ // Let PhpStan know that this loop items are not necessarily arrays
            if (!is_string($brand) and !($brand instanceof Brand)) {
                throw new ExPSAAPIException(
                    'Brand must be given either as string or Brand object, ' . gettype($brand) . ' received'
                );
            }
            if (is_string($brand)) {
                if (!Brand::isNameValid($brand) and !Brand::isCodeValid($brand)) {
                    throw new ExPSAAPIException('Invalid brand: '  . $brand);
                }
                $brand_obj = new Brand();
                if (Brand::isNameValid($brand)) {
                    $brand_obj->setName($brand); // Brand's code will be updated within the setName() method
                } else {
                    $brand_obj->setCode($brand); // Brand's name will be updated within the setCode() method
                }
                $brand = $brand_obj;
                unset($brand_obj);
            }
        }
        $this->brands = $brands;

        return $this;
    }

    public function addBrands(...$brands): self
    {
        $this->setBrands(...array_merge($this->getBrands(), $brands));

        return $this;
    }

    public function clearBrands(): self
    {
        $this->brands = [];

        return $this;
    }
}
