<?php

declare(strict_types=1);

namespace Stellantis\ExPsaApiHelper\Model;

use Stellantis\ExPsaApiHelper\Exception\ExPSAAPIException;

class Region
{
    /**
     * value → label
     */
    private const MAP = [
        'woj1' => 'dolnośląskie',
        'woj2' => 'kujawsko-pomorskie',
        'woj3' => 'lubelskie',
        'woj4' => 'lubuskie',
        'woj5' => 'łódzkie',
        'woj6' => 'małopolskie',
        'woj7' => 'mazowieckie',
        'woj8' => 'opolskie',
        'woj9' => 'podkarpackie',
        'woj10' => 'podlaskie',
        'woj11' => 'pomorskie',
        'woj12' => 'śląskie',
        'woj13' => 'świętokrzyskie',
        'woj14' => 'warmińsko-mazurskie',
        'woj15' => 'wielkopolskie',
        'woj16' => 'zachodniopomorskie',
    ];

    private ?string $value = null;
    private ?string $label = null;

    public function getValue(): ?string
    {
        return $this->value;
    }

    public function setValue(?string $value): self
    {
        $value = $this->sanitizeValue($value);
        if (!array_key_exists($value, self::MAP)) {
            // This SHOULD be a code like `woj2` or `woj14`, BUT it happens that the value stored here is the
            // same as in `label`, so: `kujawsko-pomorskie` or `warmińsko-mazurskie`, etc.
            // Check that before throwing an exception:
            foreach (self::MAP as $code => $label) {
                if ($label === $value) {
                    $this->value = $code;
                    $this->label = $label;

                    return $this;
                }
            }

            throw new ExPSAAPIException(sprintf('Invalid region value: "%s"', $value));
        }
        $this->value = $value;
        $this->label = self::MAP[$value];

        return $this;
    }

    public function getLabel(): ?string
    {
        return $this->label;
    }

    public function setLabel(?string $label): self
    {
        $label = $this->sanitizeLabel($label);
        if (!in_array($label, self::MAP)) {
            throw new ExPSAAPIException(sprintf('Invalid region label: "%s"', $label));
        }
        $this->label = $label;
        $this->value = array_flip(self::MAP)[$label];

        return $this;
    }

    /**
     * Correct some inconsistency within the external API
     */
    private function sanitizeLabel(?string $label): ?string
    {
        if (!$label) {
            return $label;
        }

        $replacements = [
            '/.*wielkopolsk.*/i' => 'wielkopolskie',
        ];

        foreach ($replacements as $pattern => $replacement) {
            $label = preg_replace($pattern, $replacement, $label);
        }

        return $label;
    }

    /**
     * Correct some inconsistency within the external API
     */
    private function sanitizeValue(?string $value): ?string
    {
        // Currently, the sanitization is exactly the same as for label:
        return $this->sanitizeLabel($value);
    }
}
