<?php

declare(strict_types=1);

namespace Stellantis\ExPsaApiHelper;

use Stellantis\ExPsaApiHelper\Exception\ExPSAAPIException;
use Stellantis\ExPsaApiHelper\Model\Brand;
use Stellantis\ExPsaApiHelper\Model\Dealer;

class Dealers extends Base
{
    public const ENDPOINT = 'Services/DealerService.svc/rest/GetDealerList';
    public const CACHE_KEY = 'dealers';

    public const QUERY_DEFAULTS = [
        'Country' => 'pl',
        'Culture' => 'pl',
        'IndicatorsActivities' => '',
        'Latitude' => '',
        'Longitude' => '',
        'RMax' => '',
        'Consumer' => 'Autobiz',
        'Details' => 'max',
        'Brand' => '',
        'Sort' => 'distance',
        'ResultMax' => '',
        'Criterias' => '',
        'Unit' => 'km',
        'CriteriasExclude' => '0',
        'GmCodeList' => 'VN',
    ];

    public function get(?Filter\Dealer $filter = null): \SplObjectStorage
    {
        $dealers = $this->fetch($filter);
        if ($filter === null) {
            return $dealers;
        }

        while ($dealers->valid()) {
            /** @var \Stellantis\ExPsaApiHelper\Model\Dealer $dealer */
            $dealer = $dealers->current();
            $dealers->next();

            if ($filter->getOnlyVN()) {
                if (!$dealer->hasVNBusiness()) { // or `!$dealer->hasNewPersonalVehiclesSalesMandate()`
                    $dealers->detach($dealer);
                    continue;
                }
            }

            if ($brands = $filter->getBrands()) {
                if (!in_array($dealer->getBrand(), $brands)) {
                    $dealers->detach($dealer);
                    continue;
                }
            }
            unset($brands);

            if ($rrdi = $filter->getRRDIs()) {
                if (!in_array($dealer->getRRDI(), $rrdi)) {
                    $dealers->detach($dealer);
                    continue;
                }
            }
            unset($rrdi);

            if ($regions = $filter->getRegions()) {
                if (!in_array($dealer->getAddress()->getRegion(), $regions)) {
                    $dealers->detach($dealer);
                    continue;
                }
            }
            unset($regions);
        }

        $dealers->rewind();

        return $dealers;
    }

    private function fetch(?Filter\Dealer $filter = null): \SplObjectStorage
    {
        $cache_key = self::CACHE_KEY;
        if ($filter) {
            $cache_key .= '_' . $filter->generateCacheKey();
        }
        $dealers = $this->loadFromCache($cache_key);

        if ($dealers === null) {
            $dealers = new \SplObjectStorage();
            foreach (Brand::NAMES as $brand_code => $brand_name) {
                // Currently only brand filter can be handled also in requests/fetching data from external API.
                // When filter contains brands, make request only for these brands:
                if ($filter && $filter_brands = $filter->getBrands()) {
                    if (!in_array($brand_code, array_map(fn(Brand $brand) => $brand->getCode(), $filter_brands))) {
                        continue;
                    }
                }
                unset($filter_brands);
                $response = $this->client->get(
                    self::ENDPOINT . '?parameters=' .
                        json_encode(
                            array_replace(
                                self::QUERY_DEFAULTS,
                                [
                                    'Brand' => $brand_code,
                                ]
                            )
                        )
                );
                try {
                    $decoded = json_decode($response->getBody()->getContents(), true, 512, JSON_THROW_ON_ERROR);
                } catch (\JsonException $exception) {
                    throw new ExPSAAPIException(
                        sprintf(
                            'Invalid data fetched from external API for brand %s: %s',
                            $brand_name,
                            $exception->getMessage()
                        )
                    );
                }
                foreach ($decoded['DealersFull'] ?? [] as $dealer_data) {
                    $dealer = (new Dealer())->hydrate($dealer_data);

                    /** @psalm-suppress InvalidArgument */
                    $dealers->attach($dealer);
                    unset($dealer);
                    unset($dealer_data);
                }
            }
            unset($brand_code, $brand_name);
            $this->saveToCache($cache_key, $dealers);
        }

        return $dealers;
    }
}
