<?php

declare(strict_types=1);

use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use Stellantis\ExPsaApiHelper\Dealers;
use PHPUnit\Framework\TestCase;
use Stellantis\ExPsaApiHelper\Model\Brand;
use Stellantis\ExPsaApiHelper\Model\Dealer;
use Stellantis\ExPsaApiHelper\Model\Address\Region;
use Stellantis\ExPsaApiHelper\Filter\Dealer as DealerFilter;

class DealersTest extends TestCase
{
    private array $history = [];

    private function createMockHandler(?array $brands = null): Dealers
    {
        // To force querying actual remote API, uncomment following line:
//        return new Dealers('https://annuaire-pdv.servicesgp.mpsa.com');

        // Create a mock using Guzzle helpers (https://docs.guzzlephp.org/en/stable/testing.html)
        $responses_queue = [];
        foreach (\Stellantis\ExPsaApiHelper\Model\Brand::NAMES as $brand_name) {
            if ($brands !== null && !in_array($brand_name, $brands)) {
                continue;
            }
            $filename = strtolower(strtr($brand_name, ['ë' => 'e']));
            $filepath = __DIR__ . DIRECTORY_SEPARATOR . 'dealers.' . $filename . '.json';
            $responses_queue[] = new Response(
                200,
                [],
                file_get_contents($filepath)
            );
        }

        $mock = new MockHandler($responses_queue);
        $handlerStack = HandlerStack::create($mock);

        $history = \GuzzleHttp\Middleware::history($this->history);
        $handlerStack->push($history);

        return new Dealers('', null, null, $handlerStack);
    }

    public function testGetNoFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get();
        $this->assertEquals(234, $result->count());
    }

    public function testGetAllForOpel()
    {
        $dealers = $this->createMockHandler([Brand::NAME_OPEL]);
        $result = $dealers->get(
            (new DealerFilter())->setBrands(Brand::CODE_OPEL)
        );
        $this->assertEquals(85, $result->count());
    }

    public function testGetAllForCitroen()
    {
        $dealers = $this->createMockHandler([Brand::NAME_CITROEN]);
        $result = $dealers->get(
            (new DealerFilter())->setBrands(Brand::CODE_CITROEN)
        );
        $this->assertEquals(66, $result->count());
    }

    public function testGetAllMultiBrand()
    {
        $dealers = $this->createMockHandler([Brand::NAME_OPEL, Brand::NAME_CITROEN]);
        $result = $dealers->get(
            (new DealerFilter())->setBrands(Brand::CODE_OPEL, Brand::NAME_CITROEN)
        );
        $this->assertEquals(151, $result->count());
    }

    public function testSingleDealerObject()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get();
        $this->assertInstanceOf(Dealer::class, $result->current());
    }

    public function testRRDIFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())->setRRDIs('440003L01X')
        );
        $this->assertEquals(1, $result->count());
    }

    public function testMultipleRRDIFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())->setRRDIs('PL0123001X', 'PLZ67K001X')
        );
        $this->assertEquals(2, $result->count());
    }

    public function testClearRRDIFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())->setRRDIs('PL0123001X', 'PLZ67K001X')->clearRRDIs()
        );
        $this->assertEquals(234, $result->count());
    }

    public function testAddRRDIFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())->setRRDIs('PL0123001X')->addRRDIs('PLZ67K001X')
        );
        $this->assertEquals(2, $result->count());
    }

    public function testNotExistingRRDIFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())->setRRDIs('foo')
        );
        $this->assertEquals(0, $result->count());
    }

    public function testRegionFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())->setRegions((new Region())->setName('podkarpackie'))
        );
        $this->assertEquals(4, $result->count());
    }

    public function testMultipleRegionFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())
                ->setRegions(
                    (new Region())->setName('łódzkie'),
                    (new Region())->setName('lubuskie')
                )
        );
        $this->assertEquals(9, $result->count());
    }

    public function testClearRegionFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())
                ->setRegions(
                    (new Region())->setName('mazowieckie'),
                )
                ->clearRegions()
        );
        $this->assertEquals(234, $result->count());
    }

    public function testAddRegionFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())
                ->setRegions(
                    (new Region())->setName('mazowieckie'),
                )
                ->addRegions((new Region())->setName('lubuskie'))
        );
        $this->assertEquals(10, $result->count());
    }

    public function testSetOnlyVNFilter()
    {
        $dealers = $this->createMockHandler();
        $result = $dealers->get(
            (new DealerFilter())->setOnlyVN(true)
        );
        $this->assertEquals(146, $result->count());
    }

    public function testCompoundFilter()
    {
        // Here brand must be set as the filter will cause (eventually mocked) HTTP client to make only 1 request
        $dealers = $this->createMockHandler([Brand::NAME_OPEL]);
        $result = $dealers->get(
            (new DealerFilter())
                ->setRegions(
                    (new Region())->setName('łódzkie'),
                )
                ->setBrands(Brand::NAME_OPEL)
        );
        $this->assertEquals(2, $result->count());
    }
}
