<?php

declare(strict_types=1);

namespace Stellantis\ExPsaApiHelper\Model\Address;

use Stellantis\ExPsaApiHelper\Exception\ExPSAAPIException;

class Region
{
    private const VALID_REGIONS = [
        'dolnośląskie',
        'kujawsko-pomorskie',
        'lubelskie',
        'lubuskie',
        'łódzkie',
        'małopolskie',
        'mazowieckie',
        'opolskie',
        'podkarpackie',
        'podlaskie',
        'pomorskie',
        'śląskie',
        'świętokrzyskie',
        'warmińsko-mazurskie',
        'wielkopolskie',
        'zachodniopomorskie',
    ];

    private ?string $name = null;

    public function setName(?string $value): self
    {
        $name = $this->sanitizeName($value);
        if (!in_array($name, self::VALID_REGIONS)) {
            throw new ExPSAAPIException(sprintf('Invalid region name: "%s"', $name));
        }
        $this->name = $name;

        return $this;
    }

    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * Correct some inconsistencies within the external API
     */
    private function sanitizeName(?string $name): ?string
    {
        if (!$name) {
            return $name;
        }

        $name = mb_strtolower($name);
        $diacritics_replacements = [
            'ą' => 'a',
            'ć' => 'c',
            'ź' => 'z',
            'ż' => 'z',
            'ś' => 's',
            'ń' => 'n',
            'ó' => 'o',
            'ł' => 'l',
            'ę' => 'e',
        ];

        $replacements = [
            '/.*poland.*/i' => 'łódzkie',
            '/.*wielkopolsk.*/i' => 'wielkopolskie',
        ];

        foreach ($replacements as $pattern => $replacement) {
            $name = preg_replace($pattern, $replacement, $name);
        }

        foreach (self::VALID_REGIONS as $valid_region) {
            if (strtr($valid_region, $diacritics_replacements) === strtr($name, $diacritics_replacements)) {
                return $valid_region;
            }
        }

        return $name;
    }
}
