<?php

declare(strict_types=1);

namespace Stellantis\ExPsaApiHelper\Model;

use Stellantis\ExPsaApiHelper\Model\Address\Coordinates;
use Stellantis\ExPsaApiHelper\Model\Address\Region;
use Stellantis\ExPsaApiHelper\Model\Dealer\Business;
use Stellantis\ExPsaApiHelper\Model\Dealer\BusinessList;
use Stellantis\ExPsaApiHelper\Model\Dealer\Email;
use Stellantis\ExPsaApiHelper\Model\Dealer\Emails;
use Stellantis\ExPsaApiHelper\Model\Dealer\Phone;
use Stellantis\ExPsaApiHelper\Model\Dealer\Phones;

class Dealer
{
    /**
     * Brand-new vehicles
     */
    public const TYPE_VN = 'VN';

    /**
     * Brand-new LCV (commercial) vehicles
     */
    public const TYPE_VL = 'VL';

    /**
     * After-sales service
     */
    public const TYPE_APV = 'APV';

    /**
     * After-sales service for LCV (commercial)
     */
    public const TYPE_CV = 'CV';

    /**
     * After-sales service for passenger (non-LCV) vehicles
     */
    public const TYPE_PC = 'PC';

    /**
     * Used (second-hand) vehicles
     */
    public const TYPE_VO = 'VO';

    private ?string $name = null;
    private ?Brand $brand = null;
    private ?string $rrdi = null;
    private ?string $siteGeo = null;
    private ?string $parentSiteGeo = null;
    private Address $address;
    private Emails $emails;
    private Phones $phones;
    private ?Coordinates $coordinates = null;
    private BusinessList $businessList;
    private string $RaisonSocial;
    private string $IntracommunityTVA;

    public function __construct()
    {
        $this->businessList = new BusinessList();
        $this->emails = new Emails();
        $this->phones = new Phones();
        $this->address = new Address();
    }

    public function getName(): ?string
    {
        return $this->name;
    }

    public function setName(?string $name): self
    {
        $this->name = $name;

        return $this;
    }

    public function getBrand(): ?Brand
    {
        return $this->brand;
    }

    public function setBrand(?Brand $brand): self
    {
        $this->brand = $brand;

        return $this;
    }

    public function getRRDI(): ?string
    {
        return $this->rrdi;
    }

    public function setRRDI(?string $rrdi): self
    {
        $this->rrdi = $rrdi;

        return $this;
    }

    public function getSiteGeo(): ?string
    {
        return $this->siteGeo;
    }

    public function setSiteGeo(?string $code): self
    {
        $this->siteGeo = $code;

        return $this;
    }

    public function getParentSiteGeo(): ?string
    {
        return $this->parentSiteGeo;
    }

    public function setParentSiteGeo(?string $code): self
    {
        $this->parentSiteGeo = $code;

        return $this;
    }

    public function getAddress(): Address
    {
        return $this->address;
    }

    public function setAddress(Address $address): self
    {
        $this->address = $address;

        return $this;
    }

    public function setBusinessList(BusinessList $businessList): self
    {
        $this->businessList = $businessList;

        return $this;
    }

    public function getBusinessList(): BusinessList
    {
        return $this->businessList;
    }

    public function addBusinesses(Business ...$businesses): self
    {
        $this->businessList->add(...$businesses);

        return $this;
    }

    public function clearBusinessList(): self
    {
        $this->businessList = new BusinessList();

        return $this;
    }

    public function hasVNBusiness(): bool
    {
        return $this->hasBusiness(self::TYPE_VN);
    }

    /**
     * Alias for `hasVNBusiness()`
     */
    public function hasNewPersonalVehiclesSalesMandate(): bool
    {
        return $this->hasVNBusiness();
    }

    public function hasBusiness(string $type): bool
    {
        return array_key_exists($type, $this->businessList->getAll());
    }

    public function getEmails(): Emails
    {
        return $this->emails;
    }

    public function setEmails(Emails $emails): self
    {
        $this->emails = $emails;

        return $this;
    }

    public function addEmail(Email ...$emails): self
    {
        foreach ($emails as $email) {
            $this->emails->add($email);
        }

        return $this;
    }

    public function clearEmails(): self
    {
        $this->emails = new Emails();

        return $this;
    }

    public function getPhones(): Phones
    {
        return $this->phones;
    }

    public function setPhones(Phones $phones): self
    {
        $this->phones = $phones;

        return $this;
    }

    public function addPhone(Phone ...$phones): self
    {
        foreach ($phones as $phone) {
            $this->phones->add($phone);
        }

        return $this;
    }

    public function clearPhones(): self
    {
        $this->phones = new Phones();

        return $this;
    }

    public function getCoordinates(): ?Coordinates
    {
        return $this->coordinates;
    }

    public function setCoordinates(?Coordinates $coordinates): self
    {
        $this->coordinates = $coordinates;

        return $this;
    }

    public function getRaisonSocial(): string
    {
        return $this->RaisonSocial;
    }

    public function setRaisonSocial(string $RaisonSocial): Dealer
    {
        $this->RaisonSocial = $RaisonSocial;

        return $this;
    }

    public function getIntracommunityTVA(): string
    {
        return $this->IntracommunityTVA;
    }

    public function setIntracommunityTVA(string $IntracommunityTVA): Dealer
    {
        // Remove rarely existing prefix `PL` from the code if it exists
        $this->IntracommunityTVA = preg_replace('/^PL(.*)/', '$1', $IntracommunityTVA);

        return $this;
    }

    public function hydrate(array $dealer_data): self
    {
        if (isset($dealer_data['Name']) and $dealer_data['Name']) {
            $this->setName(trim($dealer_data['Name']));
        }
        if (isset($dealer_data['Brand']) and $dealer_data['Brand']) {
            $this->setBrand((new Brand())->setCode($dealer_data['Brand']));
        }
        if (isset($dealer_data['Address'])) {
            $address = new Address();
            if (isset($dealer_data['Address']['City']) && $dealer_data['Address']['City']) {
                $address->setCity(trim($dealer_data['Address']['City']));
            }
            $street = implode(
                ', ',
                array_filter([
                    $dealer_data['Address']['Line1'] ?? '',
                    $dealer_data['Address']['Line2'] ?? '',
                    $dealer_data['Address']['Line3'] ?? '',
                ])
            );
            if ($street) {
                $address->setStreet($street);
            }
            unset($street);
            if (isset($dealer_data['Address']['Region']) && $dealer_data['Address']['Region']) {
                $address->setRegion((new Region())->setName(trim($dealer_data['Address']['Region'])));
            }
            if (isset($dealer_data['Address']['ZipCode']) && $dealer_data['Address']['ZipCode']) {
                $address->setZipCode(trim($dealer_data['Address']['ZipCode']));
            }

            $this->setAddress($address);
        }

        if (isset($dealer_data['BusinessList']) and $dealer_data['BusinessList']) {
            foreach ((array)$dealer_data['BusinessList'] as $business_data) {
                $this->addBusinesses(
                    new Dealer\Business(
                        $business_data['Code'],
                        $business_data['Label'],
                        $business_data['Type']
                    )
                );
            }
            unset($business_data);
        }

        if (isset($dealer_data['Emails']) and $dealer_data['Emails']) {
            foreach ((array)$dealer_data['Emails'] as $signature => $email_address) {
                if ($email_address) {
                    $this->addEmail(new Dealer\Email($signature, $email_address));
                }
            }
            unset($signature, $email_address);
        }

        if (isset($dealer_data['Phones']) and $dealer_data['Phones']) {
            foreach ((array)$dealer_data['Phones'] as $signature => $phone_number) {
                if ($phone_number) {
                    $this->addPhone(new Dealer\Phone($signature, $phone_number));
                }
            }
            unset($signature, $phone_number);
        }

        if (isset($dealer_data['RRDI']) and $dealer_data['RRDI']) {
            $this->setRRDI(trim($dealer_data['RRDI']));
        }

        if (isset($dealer_data['SiteGeo']) and $dealer_data['SiteGeo']) {
            $this->setSiteGeo(trim($dealer_data['SiteGeo']));
        }

        if (isset($dealer_data['ParentSiteGeo']) and $dealer_data['ParentSiteGeo']) {
            $this->setParentSiteGeo(trim($dealer_data['ParentSiteGeo']));
        }

        if (isset($dealer_data['Coordinates']) and $dealer_data['Coordinates']) {
            $this->setCoordinates(
                new Coordinates(
                    (float)$dealer_data['Coordinates']['Latitude'],
                    (float)$dealer_data['Coordinates']['Longitude']
                )
            );
        }

        if (isset($dealer_data['IntracommunityTVA']) and $dealer_data['IntracommunityTVA']) {
            $this->setIntracommunityTVA(trim($dealer_data['IntracommunityTVA']));
        }

        if (isset($dealer_data['RaisonSocial']) and $dealer_data['RaisonSocial']) {
            $this->setRaisonSocial(trim($dealer_data['RaisonSocial']));
        }

        return $this;
    }
}
