<?php

declare(strict_types=1);

use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use Stellantis\ExPsaApiHelper\Exception\ExPSAAPIException;
use Stellantis\ExPsaApiHelper\Model\Brand;
use Stellantis\ExPsaApiHelper\Model\Model;
use Stellantis\ExPsaApiHelper\Model\Model\Type;
use Stellantis\ExPsaApiHelper\Models;
use PHPUnit\Framework\TestCase;
use Stellantis\ExPsaApiHelper\Filter\Model as ModelFilter;

class ModelsTest extends TestCase
{
    private MockHandler $mock;
    private HandlerStack $handlerStack;
    private Models $models;

    protected function setUp(): void
    {
        parent::setUp();

        // Create a mock using Guzzle helpers (https://docs.guzzlephp.org/en/stable/testing.html)
        $this->mock = new MockHandler([
            new Response(200, [], file_get_contents(__DIR__ . DIRECTORY_SEPARATOR . 'models.json')),
        ]);
        $this->handlerStack = HandlerStack::create($this->mock);
        $this->models = new Models('', null, null, $this->handlerStack);
    }

    public function testGetNoFilter()
    {
        $result = $this->models->get();
        $this->assertEquals(80, $result->count());
    }

    public function testGetAllCitroen()
    {
        $result = $this->models->get((new ModelFilter)->setBrands(Brand::NAME_CITROEN));
        $this->assertEquals(18, $result->count());
    }

    public function testGetAllOpel()
    {
        $result = $this->models->get((new ModelFilter)->setBrands(Brand::CODE_OPEL));
        $this->assertEquals(27, $result->count());
    }

    public function testGetAllPeugeot()
    {
        $result = $this->models->get((new ModelFilter)->setBrands(Brand::NAME_PEUGEOT));
        $this->assertEquals(27, $result->count());
    }

    public function testGetAllDS()
    {
        $result = $this->models->get((new ModelFilter)->setBrands(Brand::CODE_DS));
        $this->assertEquals(8, $result->count());
    }

    public function testGetInvalidBrand()
    {
        $this->expectException(ExPSAAPIException::class);
        $this->models->get((new ModelFilter)->setBrands('foo'));
    }

    public function testSingleModelObject()
    {
        $result = $this->models->get();
        $this->assertInstanceOf(Model::class, $result->current());
    }

    public function testTypeFilter()
    {
        $result = $this->models->get((new ModelFilter)->setTypes(Type::TYPE_VP));
        while ($result->valid()) {
            $this->assertEquals(Type::TYPE_VP, $result->current()->getType());
            $result->next();
        }
        $this->assertEquals(52, $result->count());
    }

    public function testClearTypeFilter()
    {
        $result = $this->models->get((new ModelFilter)->setTypes(Type::TYPE_VP)->clearTypes());
        $this->assertEquals(80, $result->count());
    }

    public function testAddTypeFilter()
    {
        $result = $this->models->get((new ModelFilter)->setTypes(Type::TYPE_VP)->addTypes(Type::TYPE_VU));
        $this->assertEquals(80, $result->count());
    }

    public function testInvalidTypeFilter()
    {
        $this->expectException(ExPSAAPIException::class);
        $this->models->get((new ModelFilter)->setTypes('foo'));
    }

    public function testLcdv6Filter()
    {
        $result = $this->models->get((new ModelFilter)->setLcdv6('03 26'));
        $this->assertEquals(1, $result->count());
    }

    public function testAddLcdv6Filter()
    {
        $result = $this->models->get((new ModelFilter)->setLcdv6('03 26')->addLcdv6('0A 48'));
        $this->assertEquals(2, $result->count());
    }

    public function testClearLcdv6Filter()
    {
        $result = $this->models->get((new ModelFilter)->setLcdv6('03 26')->addLcdv6('0A 48')->clearLcdv6());
        $this->assertEquals(80, $result->count());
    }

    public function testModelYearsFilter()
    {
        $result = $this->models->get((new ModelFilter)->setModelYears('2022'));
        $this->assertEquals(27, $result->count());
    }

    public function testAddModelYearsFilter()
    {
        $result = $this->models->get((new ModelFilter)->setModelYears('2022')->addModelYears('2021'));
        $this->assertEquals(27, $result->count());
    }

    public function testClearModelYearsFilter()
    {
        $result = $this->models->get((new ModelFilter)->setModelYears('2022', '2021')->clearModelYears());
        $this->assertEquals(80, $result->count());
    }

    public function testVehicleNotesFilter()
    {
        $result = $this->models->get((new ModelFilter)->setVehicleNotes('2022A______'));
        $this->assertEquals(16, $result->count());
    }

    public function testAddVehicleNotesFilter()
    {
        $result = $this->models->get((new ModelFilter)->setVehicleNotes('2022A______')->addVehicleNotes('2022B______'));
        $this->assertEquals(26, $result->count());
    }

    public function testClearVehicleNotesFilter()
    {
        $result = $this->models->get(
            (new ModelFilter)->setVehicleNotes('2022A______', '2022B______')->clearVehicleNotes()
        );
        $this->assertEquals(80, $result->count());
    }

    public function testCompoundFilter()
    {
        $result = $this->models->get(
            (new ModelFilter)
                ->setBrands(Brand::NAME_CITROEN)
                ->setTypes(Type::TYPE_VU)
        );
        $this->assertEquals(5, $result->count());
    }
}
