<?php

declare(strict_types=1);

namespace Stellantis\ExPsaApiHelper\Filter;

use Stellantis\ExPsaApiHelper\Exception\ExPSAAPIException;
use Stellantis\ExPsaApiHelper\Model\Address\Region;
use Stellantis\ExPsaApiHelper\Model\Brand;
use Webmozart\Assert\Assert;
use Webmozart\Assert\InvalidArgumentException;

use function Stellantis\ExPsaApiHelper\xsort;

class Dealer
{
    use Traits\BrandsTrait;

    private array $RRDIs = [];
    private array $regions = [];
    private bool $only_vn = false;

    public function getRRDIs(): array
    {
        return $this->RRDIs;
    }

    public function setRRDIs(...$codes): self
    {
        try {
            Assert::allString($codes);
        } catch (InvalidArgumentException $exception) {
            throw new ExPSAAPIException(
                'Invalid type for RRDI code. Only strings are allowed',
                0,
                $exception
            );
        }
        $this->RRDIs = $codes;

        return $this;
    }

    public function addRRDIs(...$codes): self
    {
        $this->setRRDIs(...array_merge($this->getRRDIs(), $codes));

        return $this;
    }

    public function clearRRDIs(): self
    {
        $this->RRDIs = [];

        return $this;
    }

    public function getRegions(): array
    {
        return $this->regions;
    }

    public function setRegions(...$regions): self
    {
        $this->regions = $regions;

        return $this;
    }

    public function addRegions(...$regions): self
    {
        $this->setRegions(...array_merge($this->getRegions(), $regions));

        return $this;
    }

    public function clearRegions(): self
    {
        $this->regions = [];

        return $this;
    }

    public function getOnlyVN(): bool
    {
        return $this->only_vn;
    }

    public function setOnlyVN(bool $flag): self
    {
        $this->only_vn = $flag;

        return $this;
    }

    /**
     * Alias for `setOnlyVN()`
     */
    public function setOnlySales(bool $flag): self
    {
        return $this->setOnlyVN($flag);
    }

    /**
     * Alias for `getOnlyVN()`
     */
    public function getOnlySales(): bool
    {
        return $this->getOnlyVN();
    }

    public function generateCacheKey(): string
    {
        return md5(
            json_encode(
                [
                    'brands' => xsort(array_map(fn(Brand $brand) => $brand->getCode(), $this->brands)),
                    'RRDIs' => xsort($this->RRDIs),
                    'regions' => xsort(array_map(fn(Region $region) => $region->getName(), $this->regions)),
                    'only_vn' => $this->only_vn,
                ]
            )
        );
    }
}
