<?php

declare(strict_types=1);

namespace Stellantis\ExPsaApiHelper\Filter;

use Stellantis\ExPsaApiHelper\Exception\ExPSAAPIException;
use Stellantis\ExPsaApiHelper\Model\Model\Type;
use Webmozart\Assert\Assert;
use Webmozart\Assert\InvalidArgumentException;

class Model
{
    use Traits\BrandsTrait;

    private array $lcdv6 = [];
    private array $types = [];
    private array $model_years = [];
    private array $vehicle_notes = [];

    public function getLcdv6(): array
    {
        return $this->lcdv6;
    }

    public function setLcdv6(...$lcdv6): self
    {
        try {
            Assert::allStringNotEmpty($lcdv6);
        } catch (InvalidArgumentException $exception) {
            throw new ExPSAAPIException(
                'Invalid type for LCDV6 code. Only strings are allowed',
                0,
                $exception
            );
        }
        $this->lcdv6 = $lcdv6;

        return $this;
    }

    public function addLcdv6(...$lcdv6): self
    {
        $this->setLcdv6(...array_merge($this->getLcdv6(), $lcdv6));

        return $this;
    }

    public function clearLcdv6(): self
    {
        $this->lcdv6 = [];

        return $this;
    }

    public function getTypes(): array
    {
        return $this->types;
    }

    public function setTypes(...$types): self
    {
        try {
            Assert::allStringNotEmpty($types);
            Assert::allInArray($types, Type::getAll());
        } catch (InvalidArgumentException $exception) {
            throw new ExPSAAPIException(
                'Invalid type for... "type" ;) Valid values are: ' . join(', ', Type::getAll()),
                0,
                $exception
            );
        }
        $this->types = $types;

        return $this;
    }

    public function addTypes(...$types): self
    {
        $this->setTypes(...array_merge($this->getTypes(), $types));

        return $this;
    }

    public function clearTypes(): self
    {
        $this->types = [];

        return $this;
    }

    public function getModelYears(): array
    {
        return $this->model_years;
    }

    public function setModelYears(...$model_years): self
    {
        try {
            Assert::allNullOrNumeric($model_years);
        } catch (InvalidArgumentException $exception) {
            throw new ExPSAAPIException(
                'Model year must be an integer or null',
                0,
                $exception
            );
        }

        // Parse numerics into integers, leave null alone
        array_walk($model_years, fn(&$v) => is_null($v) ? $v = null : $v = (int)$v);

        $this->model_years = $model_years;

        return $this;
    }

    public function addModelYears(...$model_years): self
    {
        $this->setModelYears(...array_merge($this->getModelYears(), $model_years));

        return $this;
    }

    public function clearModelYears(): self
    {
        $this->model_years = [];

        return $this;
    }

    public function getVehicleNotes(): array
    {
        return $this->vehicle_notes;
    }

    public function setVehicleNotes(...$vehicle_notes): self
    {
        try {
            Assert::allStringNotEmpty($vehicle_notes);
        } catch (InvalidArgumentException $exception) {
            throw new ExPSAAPIException('Vehicle note must be non-empty string', 0, $exception);
        }
        $this->vehicle_notes = $vehicle_notes;

        return $this;
    }

    public function addVehicleNotes(...$vehicle_notes): self
    {
        $this->setVehicleNotes(...array_merge($this->getVehicleNotes(), $vehicle_notes));

        return $this;
    }

    public function clearVehicleNotes(): self
    {
        $this->vehicle_notes = [];

        return $this;
    }
}
